/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.dolphinscheduler.service.queue;

import org.apache.dolphinscheduler.common.model.Server;

import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.TimeUnit;

public class MasterPriorityQueue implements TaskPriorityQueue<Server> {

    /**
     * queue size
     */
    private static final Integer QUEUE_MAX_SIZE = 20;

    /**
     * queue
     */
    private PriorityBlockingQueue<Server> queue = new PriorityBlockingQueue<>(QUEUE_MAX_SIZE, new ServerComparator());

    private HashMap<String, Integer> hostIndexMap = new HashMap<>();

    @Override
    public void put(Server serverInfo) {
        this.queue.put(serverInfo);
        refreshMasterList();
    }

    @Override
    public Server take() throws InterruptedException {
        return queue.take();
    }

    @Override
    public Server poll(long timeout, TimeUnit unit) {
        return queue.poll();
    }

    @Override
    public int size() {
        return queue.size();
    }

    public void putList(List<Server> serverList) {
        for (Server server : serverList) {
            this.queue.put(server);
        }
        refreshMasterList();
    }

    public void remove(Server server) {
        this.queue.remove(server);
    }

    public void clear() {
        queue.clear();
        refreshMasterList();
    }

    private void refreshMasterList() {
        hostIndexMap.clear();
        Iterator<Server> iterator = queue.iterator();
        int index = 0;
        while (iterator.hasNext()) {
            Server server = iterator.next();
            hostIndexMap.put(server.getHost(), index);
            index += 1;
        }

    }

    public int getIndex(String host) {
        if (!hostIndexMap.containsKey(host)) {
            return -1;
        }
        return hostIndexMap.get(host);
    }

    /**
     * server comparator
     */
    private class ServerComparator implements Comparator<Server> {
        @Override
        public int compare(Server o1, Server o2) {
            return o1.getCreateTime().before(o2.getCreateTime()) ? 1 : 0;
        }
    }

}
