/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.gems.googlemaps;


import java.io.IOException;
import java.text.MessageFormat;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.Event;
import javax.portlet.EventRequest;
import javax.portlet.EventResponse;
import javax.portlet.PortletException;
import javax.portlet.PortletPreferences;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;

import org.apache.portals.gems.address.AddressEvent;
import org.apache.portals.gems.dojo.AbstractDojoVelocityPortlet;
import org.apache.portals.gems.dojo.DojoPortletHelper;

/**
 * This is a simple class used to override processAction
 * to save location form submission value to location preference
 *
 * @version $Id$
 */
public class GoogleMapsPortlet extends AbstractDojoVelocityPortlet
{
    
    private static final String LOAD_MAPS_JS_FUNCTION_BODY = 
        "function {0}loadMaps() '{'\r\n" +
        "  if (window.{0}mapit) '{'\r\n" +
        "    {0}mapit();\r\n" +
        "  '}' else '{'\r\n" +
        "    setTimeout(''{0}loadMaps()'', 100);\r\n" +
        "  '}'\r\n" +
        "'}'";
    
    /**
     * save submitted value
     *
     * @see javax.portlet.GenericPortlet#processActions
     *
     */
    public void processAction(ActionRequest request, ActionResponse actionResponse) throws PortletException, IOException
    {
        String location = request.getParameter( "location" );
        String mapHeight = request.getParameter( "mapheight" );
        String apiKey = request.getParameter( "apikey" );
        
        PortletPreferences preferences = request.getPreferences();
        
        if ( location != null )
        {
            preferences.setValue( "Location", location );
        }
        
        if ( mapHeight != null )
        {
            preferences.setValue( "MapHeight", mapHeight );
        }
        
        if ( apiKey != null )
        {
            preferences.setValue( "APIKey", apiKey );
        }
        
        preferences.store();
    }
    
    @Override
    protected void doHeaders(RenderRequest request, RenderResponse response)
    {
        super.doHeaders(request, response);

        String namespace = response.getNamespace();
        String loadMapsCallback = namespace + "loadMaps";
        
        String loadMapsCallbackBody = MessageFormat.format(LOAD_MAPS_JS_FUNCTION_BODY, namespace);
        DojoPortletHelper.contributeScript(response, "JavaScript", "text/javascript", null, loadMapsCallbackBody, null, null);
        
        String apiKey = request.getPreferences().getValue("APIKey", "");
        String googleMapsSourceUrl = "http://www.google.com/jsapi?key=" + apiKey + "&callback=" + loadMapsCallback;
        DojoPortletHelper.contributeScript(response, "JavaScript", "text/javascript", googleMapsSourceUrl, null, null, null);
        
        DojoPortletHelper.contributeDojoRequire(response, "dojo.lang.*");
        DojoPortletHelper.contributeDojoRequire(response, "dojo.event.*");
        DojoPortletHelper.contributeDojoRequire(response, "dojo.io.*");
        DojoPortletHelper.contributeDojoRequire(response, "dojo.widget.*");
        DojoPortletHelper.contributeDojoRequire(response, "dojo.widget.Button");
    }
    
    @Override
    public void processEvent(EventRequest request, EventResponse response) 
    {
        Event event = request.getEvent();
        if(event.getName().equals("AddressEvent"))
        {
            AddressEvent payload = (AddressEvent)event.getValue();
            response.setRenderParameter("address", payload.getAddress());
        }
    }
        
    
}
