/****************************************************************************
 * Copyright(c) 2000 Broadcom Corporation, all rights reserved
 * Proprietary and Confidential Information.
 *
 * This source file is the property of Broadcom Corporation, and
 * may not be copied or distributed in any isomorphic form without 
 * the prior written consent of Broadcom Corporation. 
 *
 * Name:        pal.h
 *
 * Description: Platform Abstracted Layer 
 *
 * Author:      Frankie Fan
 *
 * $Log:: /Source/basplnx/pal.h                                     $
 * 
 * 11    12/17/01 4:45p Ffan
 * Added support of IEEE 802.3ad Link Aggregation.
 * 
 * 10    8/03/01 9:47a Ffan
 * Cleaned up un-necessary user space resources in the spawned kernel
 * thread.
 * 
 * 9     5/24/01 6:34p Ffan
 * Added proc fs support.
 * Fixed server hang problem when the physical NIC driver is removed
 * before detaching the team.
 * 
 * 8     5/11/01 4:32p Ffan
 * Added change_mtu handler.
 * 
 * 7     1/15/01 2:50p Ffan
 * Ready for 2.4 kernel.
 * 
 * 6     1/10/01 5:44p Ffan
 * Getting ready for 2.4.0 kernel.
 * Added intercept of set_mac_address request.
 * 
 * 5     12/19/00 12:02p Ffan
 * Added NET_BH handler to send pending packets. This is necessary
 * because dev_queue_xmit() is only synchronized with bottom half.
 * 
 * 4     12/15/00 3:26p Ffan
 * 
 * 3     12/12/00 7:09p Ffan
 * Added pal_register_notifier, pal_unregister_notifier and
 * pal_stricmp functions.
 * 
 * 2     12/04/00 6:26p Ffan
 * Changed pal_device_ioctl() to be more generic.
 * Added pal_read_mii() and pal_nice_cmd() to address dependency
 * problem on linux/if.h.
 * 
 * 1     12/01/00 5:55p Ffan
 * Initial check-in.
 *
 ****************************************************************************/
#ifndef __pal_h__
#define __pal_h__

#include "bcmtype.h"

//
// macro to force unaligned data structure
//
#define __packed1__ __attribute__ ((packed, aligned(1)))

//
// Platform Abstracted Data Types
//
typedef U8* PAL_SPINLOCK;        // ptr to spinlock_t
typedef U8* PAL_ATOMIC;          // ptr to atomic_t
typedef U8* PAL_DEVICE;          // ptr to struct device
typedef U8* PAL_DEVICE_STAT;     // ptr to struct net_device_stats
typedef U8* PAL_PACKET_LIST;     // ptr to sk_buff_head
typedef U8* PAL_EVENT;           // ptr to struct wait_queue*

//
typedef U8* PAL_PACKET;          // ptr to sk_buff*
typedef U8* PAL_HANDLE;          // opaque handle
typedef unsigned long PAL_TIME;  // time data structure
typedef U8* PAL_PROC_DIR_ENTRY;  // ptr to struct proc_dir_entry

//
// Resource type definition for Platform Abstracted Data Types
//
typedef enum
{
    PRT_SPINLOCK = 0,
    PRT_ATOMIC = 1,
    PRT_DEVICE = 2,
    PRT_DEVICE_STAT = 3,
    PRT_PACKET_LIST = 4,
    PRT_BUNDLE = 5,
    PRT_EVENT = 6,
    PRT_MAXIMUM = 7
} PAL_RESOURCE_TYPE;

//
// Resource Descriptor
//
typedef struct _PAL_RESOURCE_DESCRIPTOR
{
    U32 rd_type;      // of type PAL_RESOURCE_TYPE
    U32 rd_offset;    // offset at which this resource is located
    U32 rd_size;      // size in byte of this resource
} PAL_RESOURCE_DESCRIPTOR;

//
// Cast macro
//
#define PAL_RESOURCE_CAST( _base, _prd, _type )                 \
    (_type)((_base) + (_prd)->rd_offset)

//
// PAL_NOTIFYING_EVENT
//
typedef enum
{
    PNV_UP = 1,
    PNV_DOWN = 2,
    PNV_CHANGE = 4,
    PNV_REGISTER = 5,
    PNV_UNREGISTER = 6,
    PNV_CHANGEMTU = 7,

    // make this type of size 32-bit
    PNV_CEILLING = 0xFFFFFFFF
} PAL_NOTIFYING_EVENT;

//
// PAL_NOTIFYING_FP
//
typedef int (* PAL_NOTIFYING_FP)( PAL_DEVICE dev, PAL_NOTIFYING_EVENT evt );

//
// Function prototypes
//
void 
pal_init( 
    void 
    );

void 
pal_fini( 
    void 
    );

int
pal_alloc_resource( 
    U32 res_type,
    U8** base,
    PAL_RESOURCE_DESCRIPTOR** desc,
    U32* num
    );

int
pal_alloc_bundle_resource( 
    U8** base,
    PAL_RESOURCE_DESCRIPTOR** desc,
    U32* num
    );

int
pal_free_resource(
    U8* base
    );

//
// Event functions
//
long 
pal_wait_event(
    PAL_EVENT* event,
    signed long ms
    );

void
pal_set_event(
    PAL_EVENT* event
    );

void
pal_init_event(
    PAL_EVENT* event
    );

//
// Time related functions
//
PAL_TIME
pal_get_current_time(
    void
    );

U32
pal_diff_hms(
    PAL_TIME tstart,
    PAL_TIME tend
    );

PAL_TIME
pal_advance_second(
    PAL_TIME tstart,
    unsigned long second
    );

//
// Memory copy functions
//
void*
pal_memcpy(
    void* dst,
    void* src,
    U32 len
    );

int
pal_memcmp(
    void* dst,
    void* src,
    U32 len
    );

void*
pal_memset(
    void* dst,
    int val,
    U32 len
    );

void*
pal_malloc(
    U32 len
    );

void
pal_mfree(
    void* p
    );

void
pal_copy_from_user(
    void* to,
    void* from,
    U32 len
    );

void
pal_copy_to_user(
    void* to,
    void* from,
    U32 len
    );

int
pal_stricmp(
    char* str1,
    char* str2
    );

//
// Atomic functions
//
int
pal_atomic_read(
    PAL_ATOMIC atom
    );

void
pal_atomic_set(
    PAL_ATOMIC atom,
    int val
    );

void
pal_atomic_inc(
    PAL_ATOMIC atom
    );

void
pal_atomic_dec(
    PAL_ATOMIC atom
    );

void
pal_atomic_add(
    PAL_ATOMIC atom,
    int val
    );

//
// Packet functions
//
PAL_PACKET
pal_alloc_packet(
    U32 size,
    PAL_DEVICE dev
    );

void
pal_free_packet(
    PAL_PACKET pkt
    );

void
pal_init_packet_list(
    PAL_PACKET_LIST list
    );

void
pal_enqueue_packet(
    PAL_PACKET_LIST list,
    PAL_PACKET pkt
    );

PAL_PACKET
pal_dequeue_packet(
    PAL_PACKET_LIST list
    );

void
pal_set_packet_device(
    PAL_PACKET pkt,
    PAL_DEVICE dev
    );

PAL_DEVICE
pal_get_packet_device(
    PAL_PACKET pkt
    );

void
pal_send_packet(
    PAL_PACKET pkt
    );

U8*
pal_get_packet_reserved(
    PAL_PACKET pkt
    );

U8*
pal_packet_put(
    PAL_PACKET pkt,
    unsigned int len
    );

void
pal_indicate_receive_packet(
    PAL_PACKET pkt
    );

U32
pal_get_packet_len(
    PAL_PACKET pkt
    );

U8*
pal_get_packet_data(
    PAL_PACKET pkt
    );

U8*
pal_get_packet_raw(
    PAL_PACKET pkt
    );

//
// Spinlock functions
//
void
pal_init_spinlock(
    PAL_SPINLOCK lock
    );

void
pal_acquire_spinlock(
    PAL_SPINLOCK lock,
    unsigned long* flags
    );

void
pal_release_spinlock(
    PAL_SPINLOCK lock,
    unsigned long* flags
    );

void
pal_start_bh_atomic(
    void
    );

void
pal_end_bh_atomic(
    void
    );

typedef void (*fp_hook)(void);

void
pal_hook_net_bh(
    fp_hook* orig_net_bh,
    fp_hook new_net_bh
    );

void
pal_mark_net_bh(
    void
    );

//
// Debug functions
//
void 
pal_print(
    const char* fmt,
    ...
    );

void
pal_panic(
    const char* msg
    );

int
pal_sprintf(
    char* buf,
    const char* fmt,
    ...
    );

//
// Device functions
//
int
pal_device_ioctl(
    PAL_DEVICE dev,
    void* buf,
    int buflen,
    int cmd
    );

int
pal_is_device_up(
    PAL_DEVICE dev
    );

void
pal_set_device_flags(
    PAL_DEVICE dev
    );

void
pal_set_device_tx_queue_len(
    PAL_DEVICE dev,
    int len
    );

char*
pal_get_device_name(
    PAL_DEVICE dev
    );

void
pal_set_device_name(
    PAL_DEVICE dev,
    char* name
    );

int
pal_register_device(
    PAL_DEVICE dev
    );

int
pal_unregister_device(
    PAL_DEVICE dev
    );

PAL_DEVICE
pal_get_device_by_name(
    char* name
    );

int
pal_is_device_slave(
    PAL_DEVICE dev
    );

void
pal_mark_device_slave(
    PAL_DEVICE dev,
    PAL_DEVICE master
    );

void
pal_unmark_device_slave(
    PAL_DEVICE dev,
    int hold_lock
    );

void
pal_zero_device(
    PAL_DEVICE dev
    );

void
pal_zero_device_stat(
    PAL_DEVICE_STAT stat
    );

void
pal_accumulate_device_stat(
    PAL_DEVICE dev,
    PAL_DEVICE_STAT stat
    );

void
pal_set_device_stat(
    PAL_DEVICE_STAT stat_in,
    int tx_packets,
    int rx_packets
    );

void
pal_set_device_private(
    PAL_DEVICE dev,
    void* ptr
    );

void*
pal_get_device_private(
    PAL_DEVICE dev
    );

void
pal_set_device_mac_address(
    PAL_DEVICE dev,
    U8* mac_addr
    );

U8*
pal_get_device_mac_address(
    PAL_DEVICE dev
    );

int
pal_invoke_device_set_mac_address(
    PAL_DEVICE dev,
    U8* mac_addr,
    void* set_mac_address
    );

int
pal_intercept_device_set_mac_address(
    PAL_DEVICE dev,
    void* new_set_mac_address,
    void** orig_set_mac_address
    );

int
pal_invoke_device_open(
    PAL_DEVICE dev
    );

int
pal_invoke_device_close(
    PAL_DEVICE dev
    );

void
pal_set_device_handlers(
    PAL_DEVICE dev,
    int (*init)(PAL_DEVICE),
    int (*open)(PAL_DEVICE),
    int (*stop)(PAL_DEVICE),
    int (*hard_start_xmit)(PAL_PACKET, PAL_DEVICE),
    void (*set_multicast_list)(PAL_DEVICE),
    int (*do_ioctl)(PAL_DEVICE, void*, int),
    PAL_DEVICE_STAT (*get_stats)(PAL_DEVICE),
    int (*change_mtu)(PAL_DEVICE, int)
    );

void
pal_dec_device_reference(
    PAL_DEVICE dev
    );

void
pal_inc_device_reference(
    PAL_DEVICE dev
    );

void
pal_ether_setup(
    PAL_DEVICE dev
    );

int
pal_get_device_mtu(
    PAL_DEVICE dev
    );

void
pal_set_device_mtu(
    PAL_DEVICE dev,
    int mtu
    );

int
pal_set_device_allmulticast(
    PAL_DEVICE dev,
    int inc
    );

int
pal_add_device_multicast(
    PAL_DEVICE dev,
    char* addr,
    int addr_len
    );

int
pal_del_device_multicast(
    PAL_DEVICE dev,
    char* addr,
    int addr_len
    );

int
pal_read_mii(
    PAL_DEVICE dev,
    U16 reg,
    U16* value
    );

int
pal_nice_cmd(
    PAL_DEVICE dev,
    void* buf,
    int buflen
    );

int
pal_register_notifier(
    PAL_HANDLE* pnof,
    PAL_NOTIFYING_FP fp
    );

int
pal_unregister_notifier(
    PAL_HANDLE nof
    );

//
// Thread
//
int
pal_start_thread(
    int (*fp)(void*),
    void* arg
    );

int
pal_daemonize_thread(
    char* thread_name
    );

//
// Linux module support
//   this is implemented in blf.c
//
void
pal_inc_usage(
    void
    );

void
pal_dec_usage(
    void
    );

//
// Linux kernel versions variants
//
int
pal_is_kernel_2_2(
    void
    );

int
pal_is_kernel_2_4(
    void
    );

//
// PROC FS
//
int
pal_proc_add_dir(
    char* dir
    );

int
pal_proc_remove_dir(
    char* dir
    );

int
pal_proc_add_read_entry(
    char* path,
    void* ctx
    );

int
pal_proc_remove_read_entry(
    char* path
    );

int
pal_proc_read_handler(
    char* page,
    char** start,
    off_t off,
    int count,
    int* eof,
    void* ctx
    );

typedef int (*PAL_PROC_READ_FP)( char*, char**, int, int, int*, void* );

void
pal_set_proc_read_handler(
    PAL_PROC_READ_FP fp
    );

#endif  // __pal_h__

