#!/bin/sh
# $Id: $
#
# Component:	TivoliPlus (installation utilities)
#
# $Source: $
# $Revision: $
#
# Description:  Common installation support functions for TivoliPlus modules.	
#		This script should be "sourced" at the top of the TivoliPlus
#		module initialization scripts (for ALI/server and clients).
#
# External Variable References / Initialization:
#
# Author:	Glenn Zazulia
#		Tivoli Systems, Inc.
# Modifications:
#	fpc	10/1/96	made generic
#
# (C) COPYRIGHT Tivoli Systems, Inc. 1996-1999.
# Unpublished Work
# All Rights Reserved
# Licensed Material - Property of Tivoli Systems, Inc.
#

######################################################################
#
#	IsValidClass class-name
#
IsValidClass ()
{
if wlookup -r Classes $1 >/dev/null 2>&1; then
	return 0
else
	return 1
fi
}

######################################################################
#
#	IsValidProfile profile-type profile-name
#
IsValidProfile ()
{

# make sure solaris doesn't stop here.
set +e

PRFNAME="$2"
PRFTYPE="$1"

#Check to see if profile manager exists
PM_CLASS=`wlookup -r Classes $PRFTYPE`
PRFMGR=`idlcall $PM_CLASS lookup_object '"'$PRFNAME'" {0}' 2> /dev/null`
RETCODE=$?

set -e

if [ $RETCODE -eq 0 ]; then
	return 0
else
	return 1
fi
}

######################################################################
#
#	IsValidProfileManager profile-manager-name
#
IsValidProfileManager ()
{
if IsValidProfile ProfileManager "$1"; then
	return 0
else
	return 1
fi
}

######################################################################
#
#	IsValidSentryProfile profile-name
#
IsValidSentryProfile ()
{
if IsValidProfile SentryProfile "$1"; then
	return 0
else
	return 1
fi
}

######################################################################
#
#	IsValidPresentationObject object-name
#
IsValidPresentationObject ()
{
if IsValidProfile Presentation "$1"; then
	return 0
else
	return 1
fi
}

######################################################################
#
#	IsValidFilePackageProfile profile-name
#
IsValidFilePackageProfile ()
{
if IsValidProfile FilePackage "$1"; then
	return 0
else
	return 1
fi
}

######################################################################
#
#	IsValidTaskLibrary
#
IsValidTaskLibrary ()
{
if wlookup -r TaskLibrary "$1" > /dev/null 2>&1; then
	return 0
else
	return 1
fi
}

######################################################################
#
#	IsValidTaskCollection task-library-name
#
IsValidTaskCollection ()
{
LIBRARY=`wlookup Library`
COLLECTIONGUI=`idlcall $LIBRARY lookup_object '"imp_TMF_TGC::CollectionGUI" {0}'`
COLLOBJ=`idlcall $COLLECTIONGUI lookup_object '"'$NAME'" {0}'`
if [ $? -eq 0 ]; then

	return 0
else
	return 1
fi
}

######################################################################
#
#	IsValidIndicatorCollection indicator-collection-name
#
IsValidIndicatorCollection ()
{
if wlookup -r IndicatorCollection "$1#$IRONAME" >/dev/null 2>&1; then
	return 0
else
	return 1
fi
}

######################################################################
#
#	IsValidMonitorCollection monitor-collection-name
#
IsValidMonitorCollection ()
{
if IsValidProfile MonitoringCapabilityCollection "$1" >/dev/null 2>&1; then
	return 0
else
	return 1
fi
}

######################################################################
#
#	lookup_tasklib_id
#
lookup_tasklib_id() {
  [ "$DEBUG" = true ] && set -x
  match_name="$1"
  for i in $TASKLIBS; do
    eval name='"$'${i}'"'
    if [ "$name" = "$match_name" ]; then
      echo "$i"
      return 0
    fi
  done
  return 1
}

######################################################################
#
#	get_precedence_for_level
#
get_precedence_for_level() {
  [ "$DEBUG" = true ] && set -x
  pr=0
  case $1 in
    normal_zero)                       pr=0;;
    normal        | AMS_normal)        pr=1;;
    informational | AMS_informational) pr=50;;
    warning       | AMS_warning)       pr=100;;
    severe        | AMS_severe)        pr=500;;
    critical      | AMS_critical)      pr=1000;;
    fatal         | AMS_fatal)         pr=1500;;
  esac
  echo $pr
}

######################################################################
#
#       get_state_to_copy_for_level
#
get_state_to_copy_for_level() {
  [ "$DEBUG" = true ] && set -x
  state=""
  case $1 in
    normal_zero)                       state="normal#Std_Levels.0";;
    normal        | AMS_normal)        state="normal#Std_Levels.0";;
    informational | AMS_informational) state="normal#Std_Levels.0";;
    warning       | AMS_warning)       state="warning#Std_Levels.100";;
    severe        | AMS_severe)        state="severe#Std_Levels.500";;
    critical      | AMS_critical)      state="critical#Std_Levels.1000";;
    fatal         | AMS_fatal)         state="critical#Std_Levels.1000";;
  esac
  echo $state
}

######################################################################
#
#	add_cond_levels
#
add_cond_levels() {
  [ "$DEBUG" = true ] && set -x
  prfname="$1"
  levels_to_add="$2"
  r_prfname="${prfname}#${IRONAME}"
  r_indcoll="IndicatorCollection#${IRONAME}"

  for lvl in $levels_to_add; do
    precedence=`get_precedence_for_level $lvl`
    if wdellevel $lvl "$r_prfname"; then
      log_info "$lvl level removed for $r_prfname"
    fi
    if waddlevel $lvl $precedence "$r_prfname" > /dev/null; then
      log_success "$lvl level added for $r_prfname, precedence $precedence)"
      #
      # add a new state to the IndicatorCollection for this level
      #
      state_to_copy=`get_state_to_copy_for_level $lvl`
      wgetbitmap -I "$state_to_copy" -r "$r_indcoll" > ${TEMP}/newicon.xpm
      wgetdialog -I "$state_to_copy" -r "$r_indcoll" > ${TEMP}/newdialog.d
      if eval wputicon -r "$r_indcoll" '"'$lvl'#Std_Levels.'$precedence'"' \
		  ${TEMP}/newdialog.d ${TEMP}/newicon.xpm > /dev/null; then
         log_success "$lvl state added to IndicatorCollection"
      fi
      rm ${TEMP}/newicon.xpm ${TEMP}/newdialog.d
    fi
  done
}

######################################################################
#
#	normalize_name
#
normalize_name() {
  echo "$1" | sed 's/[^a-zA-Z0-9]/_/g'
}

######################################################################
#
#	roundoff_rate
#
roundoff_rate() {
  [ "$DEBUG" = true ] && set -x
  if [ "$2" = seconds ]; then
    mins=0
    secs=0
    if expr $1 / 60 > /dev/null 2>& 1; then
      mins=`expr $1 / 60`
    fi
    if expr $1 % 60 > /dev/null 2>& 1; then
      secs=`expr $1 % 60`
    fi
    if [ $secs -gt 30 ]; then
      mins=`expr $mins + 1`
    fi
    echo "$mins minutes"
  else
    echo "$1 $2"
  fi  
}  

######################################################################
#
#	is_predefined_dep_task
#
is_predefined_dep_task()
{

    [ "$DEBUG" = true ] && set -x

    case "$1" in
      "File Version")			return 0;;
      "File Excluded")			return 0;;
      "Required Swap Space")		return 0;;
      "Required Memory")		return 0;;
      "Required Disk Space")		return 0;;
      ".INI File Entry Excluded")	return 0;;
      "File Existence")			return 0;;
      ".INI File Entry")		return 0;;
      "OS Version")			return 0;;
      "CPUarchitecture")		return 0;;
      "Registry Entry")			return 0;;
      "Registry Entry Excluded")	return 0;;
    esac
    return 1
}

######################################################################
#
#       check_tme_server
#
check_tme_server()
{
  [ "$DEBUG" = true ] && set -x
  MNOID=`objcall 0.0.0 get_host_location`
  ALI_OID=`wlookup -r distinguished ServerManagedNode`
  if [ "$ALI_OID" = "$MNOID" ]; then
        return 0
  else
        return 1
  fi
}

######################################################################

log_echo ()
{
echo "$*" >>$TEMP/${SHORT_PRODNAME}_PLUS.log
}

log_failure ()
{
log_echo "--- "$*""
}

log_success ()
{
log_echo "+++ "$*""
}

log_info ()
{
log_echo "    "$*""
}

log_start_install ()
{
log_echo "***** Start Install: $MEDIUM_PRODNAME *****"
log_echo "***** `date` *****"
}

log_start_action ()
{
LAST_ACTION="$*"
log_echo ">>> $*"
}

log_end_action ()
{
if [ $# -eq 0 ]; then
	msg=$LAST_ACTION
else
	msg="$*"
fi
log_echo "<<< $msg complete!"
}

log_end_install ()
{
log_echo "***** `date` *****"
log_echo "***** End Install: $MEDIUM_PRODNAME *****"
}

errecho()
{
echo "$*" >&2
}

######################################################################
#
# check_apps_installed
#
# Checks which applications are installed
# Exports SENTRY_INSTALLED, COURIER_INSTALLED, TEC_INSTALLED, etc.
#	(set to true or false)
#
check_apps_installed()		# check_apps_installed <defsentry> [check_mon]
{

[ "$DEBUG" = true ] && set -x

#
#	See if Sentry is installed
#
#-#-#
log_start_action "Determine if Sentry is installed"

if IsValidClass SentryProfile; then
	SENTRY_INSTALLED=true
	#-#-#
	log_success "Sentry is installed"

	REGION_LEN=`odadmin | grep Region | awk '{print $3}' | wc -c`

	#Check to see if the required sentry functionality is present
	# only do this once, because there's a bug that causes it to
	# fail if you do it a second time in close proximity to the first
	if [ -z "$SENTRY_MONITORS_OK" ]; then
		if eval wlsmon -s '"'$DEFSEN'"' >/dev/null; then
			SENTRY_MONITORS_OK=true
		else
			SENTRY_MONITORS_OK=false
		fi
	fi
else
	SENTRY_INSTALLED=false
	#-#-#
	log_failure "Sentry is NOT installed"
fi
#-#-#
log_end_action

#
#	See if Courier is installed
#
#-#-#
log_start_action "Determine if Courier is installed"

if IsValidClass FilePackage; then
	#-#-#
	log_success "Courier is installed"
	COURIER_INSTALLED=true
else
	#-#-#
	log_failure "Courier is NOT installed"
	COURIER_INSTALLED=false
fi
#-#-#
log_end_action

#
#	See if the TEC is installed
#
#-#-#
log_start_action "Determine if TEC is installed"

if IsValidClass EventServer; then
	#-#-#
	log_success "TEC is installed"
	TEC_INSTALLED=true
else
	#-#-#
	log_failure "TEC is NOT installed"
	TEC_INSTALLED=false
fi
#-#-#
log_end_action

#
#	See if the logfile adapter is installed
#
#-#-#
log_start_action "Determine if TEC Logfile adapter is installed"

TECAD_DIR=$BINDIR/TME/TEC/adapters
if [ ! -d $TECAD_DIR ]; then
  TECAD_DIR=$BINDIR/../../lcf/TME/TEC/adapters
fi

if [ x\"$OS\" = x\"Windows_NT\" ] ; then
	TECAD_EXE=$TECAD_DIR/bin/tecad_nt.exe
else
	TECAD_EXE=$TECAD_DIR/bin/tecad_logfile
fi
if [ -f $TECAD_EXE ]; then
	#-#-#
	log_success "TEC Logfile adapter is installed"
	TECAD_LOGF_INSTALLED=true
else
	#-#-#
	log_failure "TEC Logfile adapter is NOT installed"
	TECAD_LOGF_INSTALLED=false
fi
#-#-#
log_end_action
}

######################################################################
#
# add_res_bitmaps
#
add_res_bitmaps()
{

[ "$DEBUG" = true ] && set -x

if [ $# -ne 1 ]; then
	errecho "No resource specified."
	return 1
fi
RESOURCE=$1

wputbitmap -r "$RESOURCE" SLINK < $LINKDIR/SLink_icon.xpm  > /dev/null 2>&1
wputbitmap -r "$RESOURCE" CLINK < $LINKDIR/CLink_icon.xpm > /dev/null 2>&1
wputbitmap -r "$RESOURCE" NCLINK < $LINKDIR/NCLink_icon.xpm > /dev/null 2>&1
wputbitmap -r "$RESOURCE" SULINK < $LINKDIR/SULink_icon.xpm > /dev/null 2>&1
wputbitmap -r "$RESOURCE" JLINK < $LINKDIR/JLink_icon.xpm > /dev/null 2>&1
wputbitmap -r "$RESOURCE" TSPM < $LINKDIR/PM_icon.xpm > /dev/null 2>&1
wputbitmap -r "$RESOURCE" TSPMD < $LINKDIR/PM_icon_dataless.xpm > /dev/null 2>&1
wputbitmap -r "$RESOURCE" TEC < $LINKDIR/TEC_icon.xpm > /dev/null 2>&1
wputbitmap -r "$RESOURCE" LOG < $LINKDIR/LOG_icon.xpm > /dev/null 2>&1
}

######################################################################
#
# install_link_class
#
install_link_class()
{

[ "$DEBUG" = true ] && set -x

# Install the SolutionsLink Class if not already installed
#-#-#
log_start_action "Installing Link Class"
if IsValidClass SolutionsLink; then
	echo Link installed
else
	Link.install
fi
#-#-#
log_success "Link Class installed"

#-#-#
log_end_action

# This should be done as part of Link install...
add_res_bitmaps Job
add_res_bitmaps TaskLibrary

if IsValidClass EnterpriseClient; then
	wputbitmap -r EnterpriseClient TivoliPlus <$LINKDIR/Sol_icon.xpm  > /dev/null 2>&1
	wputbitmap -r EnterpriseClient JLINK <$LINKDIR/JLink_icon.xpm   > /dev/null 2>&1
fi
}


######################################################################
#
# create_task
#
# Sets up task, job, dialog, etc.
# See below for defaults.
# External dependencies:	R_TASKLIB, JOB_PM, MNNAME, PRNAME
#
create_task()	# -t <task_nm> -m <method> [-d <dialog>] [-r <role>] [-u <user>]
		#[-M <job mode>] [-T <job timeout>] [-p <pm>] [-h <mannode>]
		#[-i <icon state>] [-taskonly]
{
[ "$DEBUG" = true ] && set -x
#
# Variable initializations / defaults:
#
role=admin
user="\$root_user"
mode=parallel
timeout=0
taskonly=false
tasklib=$DEFAULT_TASKLIB

set +e
unset	task method dialog dest notarget # collection
unset	iconstate		# Choices:  SLINK CLINK NCLINK SULINK JLINK
set -e

#
# Parse arguments:
#
#		-c*) collection=$2;		shift	;;	# collection name

while [ $# -gt 0 ]; do
	case $1 in
		-taskonly) taskonly=true		;;	# Skip remaining setup
		-t*) task="$2";		shift	;;	# Task name
		-m*) method=$2;		shift	;;	# Method name
		-d*) dialog=$2;		shift	;;	# Dialog descriptor
		-r*) role=$2;		shift	;;	# Execution role
		-u*) user=$2;		shift	;;	# Execution user
		-M*) mode=$2;		shift	;;	# Job mode
		-T*) timeout=$2;	shift	;;	# Job timeout
		-i*) iconstate=$2;	shift	;;	# Icon state
		-j*) jobname=$2;	shift	;;	# Job name
		-l*) tasklib=$2;	shift	;; # Task library name
		-h*) dest="$dest -h \"$2#$IRONAME\"";	shift	;; # Job target mannode
		-p*) dest="$dest -p \"$2#$IRONAME\"";	shift	;; # Job target profmgr
		-n*) notarget=1				;;	# No Job target
		*)
			errecho "create_task:  '$1' invalid."
			return 1
			;;
	esac
	if [ $# -lt 1 ]; then
		errecho "create_task:  syntax error:  $*"
		return 1
	fi
	shift
done

log_info "task:        $task"
log_info "method:      $method"
log_info "dialog:      $dialog"
log_info "role:        $role"
log_info "user:        $user"
log_info "mode:        $timeout"
log_info "iconstate:   $iconstate"
log_info "tasklib:     $tasklib"
log_info "jobname:     $jobname"
log_info "dest:        $dest"
log_info "notarget:    $notarget"

if [ -z "$task" -o -z "$method" ]; then
	errecho "create_task:  must specify task and method names."
	return 1
fi

if is_predefined_dep_task "$task"; then
	log_info "$task is a predefined dependency task ... skipping"
        return 0
fi

[ -z "$dest$notarget" -a -n "$JOB_PM" ] && dest="-p \"$JOB_PM#$IRONAME\""

if [ "$user" = "$NULL" ]; then
	user=\$root_user
fi

r_tasklib="${tasklib}#${IRONAME}"

#
# Set up task
#
if [ "$USE_TLL" = true ]; then
	task=`normalize_name "$task"`   # strip blanks, since builder would have
fi

#
# force the wlstlib's to run in the C locale
#
USER_LANG=$LANG
LANG=C
export LANG

if wlstlib "$r_tasklib" | grep "^(task)	$task\$" >/dev/null 2>&1; then
	cmd=wsettask
else
	cmd=wcrttask
fi
$cmd -t "$task" -l "$r_tasklib" -r "$role" -u "$user" -i default "$MNNAME" "$method"
$taskonly && return 0

#
# Skip creating a plus icon for the predefined Product_Install task, it wouldn't make sense
#
if [ "$task" = "Product_Install" ]; then
  return 0
fi

#
# Set up job
#

# Make sure job name is unique-ified
task_job="`echo $task | sed s/\" ($SHORT_PRODNAME)$\"//` ($SHORT_PRODNAME)"

if wlstlib "$r_tasklib" | grep "^(job)	$task_job\$" >/dev/null 2>&1; then
	cmd=wsetjob
	#-#-#
	log_info "$task_job exists, overriding settings"
else
	cmd=wcrtjob
	#-#-#
	log_info "$task_job does NOT exist, creating"
fi
eval $cmd -j '"$task_job"' -l '"$r_tasklib"' -t '"$task"' -M '"$mode"' -m $timeout -D -o 15 $dest

#
# restore LANG 
#
LANG=$USER_LANG
export LANG

#
# Finish
#
if [ "$dialog" = "$NULL" ]; then
	dialog=
fi

if [ -n "$dialog" -a ! -f "$dialog" ]; then
	log_info "Could not find dialog $dialog ... using default"
fi
if [ -f "$dialog" ]; then
	#-#-#
	log_info "adding dialog to $task"
#	wputdialog -l @TaskLibrary:"$DEFAULT_TASKLIB" "$task" >/dev/null 2>&1 < $dialog
	wputdialog -l @TaskLibrary:"$r_tasklib" "$task" >/dev/null 2>&1 < $dialog
	wputdialog -l @TaskLibrary:"$r_tasklib" "$task_job" >/dev/null 2>&1 < $dialog
fi

wcrtjlink "$MEDIUM_PRODNAME" "$tasklib" "$task" "$jobname" "$task_job" $iconstate
#wcrtjlink "$collection" "$DEFAULT_TASKLIB" "$task" "$task" $iconstate
#-#-#
log_success "$task installed"
}

######################################################################
#
# create_app_launch
#
create_app_launch()
{

[ "$DEBUG" = true ] && set -x

if [ -n "$LAUNCH_ICONS" ]; then
  if [ -z "$STDBITMAP" ]; then
    STDBITMAP="$PRODDIR/Default.xpm"
  fi
  if [ ! -f "$STDBITMAP" ]; then
    #-#-#
    log_failure "cannot create launch icon: STDBITMAP file $STDBITMAP does not exist"
    return 1
  fi
fi

for icon in $LAUNCH_ICONS; do
	eval LAUNCHNAME='"$'$icon'"'
	eval LAUNCHDSL='"$'$icon'_DSL"'
	eval LAUNCHMENU='"$'$icon'_MENUS"'
	eval LAUNCHDEPS='"$'$icon'_DEPS"'

	#-#-#
	log_info "checking launch icon dependencies"

	if check_dependencies $LAUNCHDEPS; then
		#-#-#
		log_start_action "Create AppLaunch icon $LAUNCHNAME"

		#-#-#
		log_info "menu dialog: $LAUNCHDSL"

		MENUITEMS=""

		for menu in $LAUNCHMENU; do
			eval MENUNAME='"$'$menu'"'
			eval MENUSCRIPT='"$'$menu'_SCRIPT"'

			#-#-#
			log_info "menu choice: $MENUNAME"
			#-#-#
			log_info "menu script: $PRODDIR/$MENUSCRIPT"

			MENUITEMS="$MENUITEMS $MENUSCRIPT $PRODDIR/$MENUSCRIPT "
		done

		cfgappicon "$MEDIUM_PRODNAME" "$LAUNCHNAME" $STDBITMAP $LAUNCHDSL \
			$MENUITEMS

		# add any extra dialogs to the presentation object
		for menu in $LAUNCHMENU; do
			eval MENUNAME='"$'$menu'"'
			eval MENUDIALOG='"$'$menu'_DIALOG"'
			if [ -n "$MENUDIALOG" -a -f $PRODDIR/$MENUDIALOG ]; then
				# install the callback dialog
				wputdialog -r "$LAUNCHNAME"_"$MNNAME"_"$MEDIUM_PRODNAME" \
					$MENUDIALOG >/dev/null 2>&1 < $PRODDIR/$MENUDIALOG
				# install it's required bitmap
				wputbitmap -r "$LAUNCHNAME"_"$MNNAME"_"$MEDIUM_PRODNAME" \
					APPICON <$STDBITMAP > /dev/null 2>&1
			fi
		done
	
		#-#-#
		log_success "$LAUNCHNAME installed"
	fi

	#-#-#
	log_end_action
done
}

######################################################################
#
# create_subscription_lists
#
create_subscription_lists()
{

[ "$DEBUG" = true ] && set -x

for sublist in $SUBLIST_PMS; do
	eval SUBLIST='"$'$sublist'"'
	eval SUBLIST_ICONDSL='"$'$sublist'_ICONDSL"'
	eval SUBLIST_ICONSTATE='"$'$sublist'_ICONSTATE"'

	#-#-#
	log_start_action "Create Subscription List $SUBLIST"

	if [ -n "$SUBLIST_ICONDSL" -a -f $PRODDIR/$SUBLIST_ICONDSL ]; then
		wputicon -r ProfileManager $SUBLIST_ICONSTATE	\
						$PRODDIR/$SUBLIST_ICONDSL \
						$LINKDIR/PM_icon.xpm
		wcrtsprfmgr "$MEDIUM_PRODNAME" "$SUBLIST" $SUBLIST_ICONSTATE
	else
		if [ -n "$SUBLIST_ICONDSL" ]; then
		  log_info "Could not find $SUBLIST_ICONDSL ... using default"
		fi
		wcrtsprfmgr "$MEDIUM_PRODNAME" "$SUBLIST"
	fi


	#-#-#
	log_end_action
done
}

######################################################################
#
# sublist_task_name
#
sublist_task_name()
{
  echo "Create_Subscriber_List"
}

######################################################################
#
# create_sublist_task
#
create_sublist_task()
{

[ "$DEBUG" = true ] && set -x

	if $LINKDIR/PLUStme_server.sh; then
		#-#-#
		log_start_action "Create Subscriber List Task"

		TASK_NAME="`sublist_task_name`"
		TASK_ICON=$LINKDIR/SULink_icon.xpm
		TASK_METHOD=$LINKDIR/PLUScreate_sublist_task.sh
		TASK_DIALOG=$LINKDIR/PLUScreate_sublist_task_sh.d
		TASK_ICONDSL=$LINKDIR/PLUScreate_sublist_task_icon.d
		TASK_ICONSTATE=STATE_PLUScreate_sublist_task_icon.d
		TASK_USER=$NULL

		LINK_CO=`wlookup -r Classes SolutionsLink`
		initpres $LINK_CO -class -put \
			state "$TASK_ICONSTATE" $TASK_ICONDSL \
				xpm $TASK_ICON
	
		create_task \
			-t "$TASK_NAME" \
			-m $TASK_METHOD \
			-d $TASK_DIALOG \
			-h "$MNNAME" \
			-r senior \
			-u $TASK_USER \
			-M parallel \
			-T 60 \
			-j "$TASK_NAME" \
			-i $TASK_ICONSTATE

		#-#-#
		log_end_action
	fi
}

######################################################################
#
# add_host_to_subscription_lists
#
add_host_to_subscription_lists()
{ 

[ "$DEBUG" = true ] && set -x

for sublist in $SUBLIST_PMS; do
	eval R_SUBLIST='"$R_'$sublist'"'
	eval SUBLIST_DEPS='"$'$sublist'_DEPS"'

	#-#-#
	log_start_action "Add host $MNNAME to Subscription list $R_SUBLIST"

	#
	# if there's a subscription list validation script, unsubscribe
	#	the host first, then check the validity and re-subscribe
	#	if appropriate
	#
	if [ -z "$SUBLIST_DEPS" ]; then
		wsub @ProfileManager:"$R_SUBLIST" @ManagedNode:$MNNAME#$IRONAME
	else
		for dep in $SUBLIST_DEPS; do
			eval DEPSCRIPT='"$'$dep'"'

			if [ -f "$PRODDIR/$DEPSCRIPT" ]; then
				wunsub -l @ProfileManager:"$R_SUBLIST" @ManagedNode:$MNNAME#$IRONAME
				#-#-#
				log_info "Executing dependency script $DEPSCRIPT"

				set +e
				$PRODDIR/$DEPSCRIPT
				DEPRETCODE=$?
				set -e

				if [ $DEPRETCODE -eq 0 ]; then
					#-#-#
					log_success "Subscribing $MNNAME to $R_SUBLIST"
					wsub @ProfileManager:"$R_SUBLIST" @ManagedNode:$MNNAME#$IRONAME
				else
					#-#-#
					log_failure "$MNNAME does not satisfy dependency requirement"
				fi
			fi
		done
	fi
	#-#-#
	log_end_action
done
}

######################################################################
#
# check_dependencies
#
check_dependencies()
{

[ "$DEBUG" = true ] && set -x
#
# save the arguments locally first, to prevent hpux9s from losing them
#
DEPSCRIPTS=""
for arg in $*; do
	eval SCRIPTNAME='"$'$arg'"'
	DEPSCRIPTS="$DEPSCRIPTS $SCRIPTNAME"
done

if [ -n "$DEPSCRIPTS" ]; then
	#-#-#
	log_info "Checking dependencies"
else
	#-#-#
	log_info "No dependencies specified"
	return 0
fi

for script in $DEPSCRIPTS; do
	if [ -f "$PRODDIR/$script" ]; then
		#-#-#
		log_info "Executing dependency script $script"

		set +e
		$PRODDIR/$script > /dev/null
		DEPRETCODE=$?
		set -e

		if [ $DEPRETCODE -ne 0 ]; then
			#-#-#
			log_failure "Failed dependency"
			return 1
		fi
	fi
done

	#-#-#
log_success "Passed dependency check"
return 0
}

######################################################################
#
# create_task_collections
#
create_task_collections()
{
#
#	Always create this one.
#
#-#-#
log_start_action "Create TaskLibrary $DEFAULT_TASKLIB"
if IsValidTaskLibrary "$DEFAULT_TASKLIB"; then
	#-#-#
	log_success "$DEFAULT_TASKLIB exists";
else
	if wcrttlib "$DEFAULT_TASKLIB" "$MEDIUM_PRODNAME#$IRONAME"; then
		#-#-#
		log_success "$DEFAULT_TASKLIB created"
	else
		#-#-#
		log_failure "Failed to create $DEFAULT_TASKLIB"
	fi
fi

#-#-#
log_end_action

#
#	Create the rest.
#
for tasklib in $TASKLIBS; do
	eval TASKLIB='"$'$tasklib'"'
	#-#-#
#	log_start_action "Create TaskCollection $TASKLIB"
	log_start_action "Create TaskLibrary $TASKLIB"
	if IsValidTaskLibrary "$TASKLIB"; then
		#-#-#
		log_success "$TASKLIB exists"
	else
		if wcrttlib "$TASKLIB" "$MEDIUM_PRODNAME#$IRONAME" 2>/dev/null; then
			#-#-#
			log_success "$TASKLIB created"
		else
			#-#-#
			log_failure "Failed to create $TASKLIB"
			return 1
		fi
	fi
	#-#-#
	log_end_action
done
}

######################################################################
#
# create_tasks
#
create_tasks ()
{

[ "$DEBUG" = true ] && set -x

for task in $TASK_LIST; do
	eval TASKNAME='"$'$task'"'
	eval TASKJOB='"$'$task'_JOB"'
	eval TASKSCRIPT='"$'$task'_SCRIPT"'
	eval TASKUSER='"$'$task'_USER"'
	eval TASKDIALOG='"$'$task'_DIALOG"'
	eval TASKLIBR='"$'$task'_LIBR"'
	eval TASKDEPENDS='"$'$task'_DEPS"'
	eval TASKICONDSL='"$'$task'_ICONDSL"'
	eval TASKICONSTATE='"$'$task'_ICONSTATE"'
	eval TASKTARGETS='"$'$task'_TARGETS"'

	# _about and _create_sublist tasks should only be created on the TMR server
	if [ "$TASKSCRIPT" = "_about.sh" -o "$TASKSCRIPT" = _create_sublist.sh ]; then
		if $LINKDIR/PLUStme_server.sh; then
			TASKTARGETS=""
		else
			return
		fi
	fi

	#-#-#
	log_start_action "Create $TASK_TYPE Task $TASKNAME"

	# go ahead and expand $TASKICONDSL and $TASKDIALOG
	TASKICONDSL=$PRODDIR/$TASKICONDSL
	if [ "$TASKSCRIPT" = "_create_sublist.sh" ]; then
		TASKDIALOG=$LINKDIR/PLUScreate_sublist_task_sh.d
		TASKICONDSL=$LINKDIR/PLUScreate_sublist_task_icon.d
	elif [ -z "$TASKDIALOG" -o ! -f $PRODDIR/$TASKDIALOG ] ; then
		TASKDIALOG=$NULL
	else
		TASKDIALOG=$PRODDIR/$TASKDIALOG
	fi

	if [ -z "$TASKLIBR" ] ; then
		TASKLIBR=$DEFAULT_TASKLIB
	fi

	if [ -z "$TASKUSER" ] ; then
		TASKUSER=$NULL
	fi

	if [ ! -f $TASKDIALOG ]; then
		log_info "Could not find $TASKDIALOG ... using default"
	fi

	if [ ! -f $TASKICONDSL ]; then
		log_info "Could not find $TASKICONDSL ... using default"
		TASKICONSTATE=JLINK
	else
		LINK_CO=`wlookup -r Classes SolutionsLink`
		initpres $LINK_CO -class -put \
			state "$TASKICONSTATE" $TASKICONDSL \
				xpm $LINKDIR/$TASK_ICON
	fi
	
	if check_dependencies $TASKDEPENDS; then
		if [ -z "$TASKTARGETS" ]; then
			create_task \
				-t "$TASKNAME" \
				-m $PRODDIR/$TASKSCRIPT \
				-d $TASKDIALOG \
				-h "$MNNAME" \
				-i "$TASKICONSTATE" \
				-j "$TASKJOB" \
				-l "$TASKLIBR" \
				-r admin \
				-u $TASKUSER \
				-M parallel \
				-T 0
		else
			TARGETS=""
			for target in $TASKTARGETS; do
				eval TARG='"$'$target'"'
				TARGETS="$TARGETS -p \"$TARG\""
			done

			eval create_task \
				-t \"$TASKNAME\" \
				-m $PRODDIR/$TASKSCRIPT \
				-d $TASKDIALOG \
				$TARGETS \
				-i \"$TASKICONSTATE\" \
				-j \"$TASKJOB\" \
				-l \"$TASKLIBR\" \
				-r admin \
				-u $TASKUSER \
				-M parallel \
				-T 0
		fi
	fi

	#-#-#
	log_end_action
done
}

######################################################################
#
# create_administrative_tasks
#
create_administrative_tasks ()
{
[ "$DEBUG" = true ] && set -x
TASK_LIST="$TASKS"
TASK_TYPE="Administrative"
TASK_ICON=JLink_icon.xpm

	#-#-#
	log_start_action "Create Administrative Task $TASKNAME"

if [ "$USE_TLL" = true ]; then
	compile_tll_for_tasks $TASK_LIST
fi
create_tasks

	#-#-#
	log_end_action
	
}

######################################################################
#
# create_plus_config_tasks
#
create_plus_config_tasks ()
{
[ "$DEBUG" = true ] && set -x
TASK_LIST="$CONFIGTASKS $ABOUTTASKS"
TASK_TYPE="Plus Configuration"
TASK_ICON=SULink_icon.xpm

	#-#-#
	log_start_action "Create Plus Config Task $TASKNAME"

if [ "$USE_TLL" = true ]; then
	compile_tll_for_tasks $TASK_LIST
fi
create_tasks

	#-#-#
	log_end_action
	
}

######################################################################
#
# create_tec_config_tasks
#
create_tec_config_tasks ()
{
[ "$DEBUG" = true ] && set -x
TASK_LIST="$TECTASKS"
TASK_TYPE="TEC Configuration"
TASK_ICON=TEC_icon.xpm

	#-#-#
	log_start_action "Create TEC Config Task $TASKNAME"

if [ "$USE_TLL" = true ]; then
	compile_tll_for_tasks $TASK_LIST
fi
create_tasks

	#-#-#
	log_end_action
	
}

######################################################################
#
# create_log_config_tasks
#
create_log_config_tasks ()
{
[ "$DEBUG" = true ] && set -x
TASK_LIST="$LOGTASKS"
TASK_TYPE="Logfile Adapter Configuration"
TASK_ICON=LOG_icon.xpm

	#-#-#
	log_start_action "Create Logfile Adapter Config Task $TASKNAME"

if [ "$USE_TLL" = true ]; then
	compile_tll_for_tasks $TASK_LIST
fi
create_tasks

	#-#-#
	log_end_action
	
}

######################################################################
#
# create_sentry_hidden_tasks
#
create_sentry_hidden_tasks ()
{
[ "$DEBUG" = true ] && set -x
TASK_LIST="$SENTRYTASKS"

compile_tll_for_tasks $TASK_LIST
create_hidden_tasks

}

######################################################################
#
# create_tec_hidden_tasks
#	This creates Hidden tasks for Sentry and T/EC.
#
create_tec_hidden_tasks ()
{
[ "$DEBUG" = true ] && set -x
TASK_LIST="$HIDDENTASKS"

compile_tll_for_tasks $TASK_LIST
create_hidden_tasks

}

######################################################################
#
# create_hidden_tasks
#	This creates Hidden tasks for Sentry and T/EC.
#
create_hidden_tasks ()
{

[ "$DEBUG" = true ] && set -x

for task in $TASK_LIST; do
	eval TASKNAME='"$'$task'"'
	eval TASKSCRIPT='"$'$task'_SCRIPT"'
	eval TASKUSER='"$'$task'_USER"'
	eval TASKDEPENDS='"$'$task'_DEPS"'
	eval TASKLIBR='"$'$task'_LIBR"'

	if [ -z "$TASKUSER" ] ; then
		TASKUSER=$NULL
	fi

	if [ -z "$TASKLIBR" ] ; then
		TASKLIBR=$DEFAULT_TASKLIB
	fi

	#-#-#
	log_start_action "Create Hidden Task $TASKNAME"

	if check_dependencies $TASKDEPENDS; then
		create_task \
			-t "$TASKNAME" \
			-m $PRODDIR/$TASKSCRIPT \
			-r admin \
			-u $TASKUSER \
			-l "$TASKLIBR" \
			-taskonly
	fi
	#-#-#
	log_end_action
done
}

######################################################################
#
# compile_tll_for_tasks
#
compile_tll_for_tasks()
{
  [ "$DEBUG" = true ] && set -x

  # first compile all tll files associated with the tasks
  processed_libs=""
  for task in $*; do
    task_id="$task"
    eval taskdeps='"$'${task_id}'_DEPS"'
    eval libname='"$'${task_id}'_LIBR"'
    if [ -z "$libname" ] ; then
	libname="$DEFAULT_TASKLIB"
    fi
    r_libname="${libname}#${IRONAME}"
    if lookup_tasklib_id "$libname"; then
      lib_id=`lookup_tasklib_id "$libname"`
      eval tll_file='"$'${lib_id}'_TLL"'
    fi

    #-#-#
    log_start_action "Create TLL Tasks in TaskLibrary $libname"

    if check_dependencies $taskdeps; then

      # if a tll file is specified and we haven't already, then compile it now!
      if [ "${tll_file}" ]; then
        found=false
        for j in $processed_libs; do
          if [ "$j" = "$lib_id" ]; then
            found=true
          fi
        done
        if $found; then
          :
        else  
          processed_libs="$lib_id $processed_libs"
          if [ -f ${PRODDIR}/${tll_file} ]; then
	    cpp_cmd=`cpp_path`
	    if [ "$cpp_cmd" ]; then
	      cpp_opt="-P"    # tell cpp to suppress next-phase directives 
	    else
	      cpp_cmd=cat
	      cpp_opt=""
	    fi
            tmp_tll=${TEMP}/${tll_file}
            log_info "       adding all tasks defined in \"$tll_file\""
            # remove template comments, just in case cat is our preprocessor
            sed s/"TaskLibrary.*"/"TaskLibrary \"${libname}\""/ "${PRODDIR}/${tll_file}" \
		| grep -v '^.\*' > ${tmp_tll}
            wtll -r -p "${MEDIUM_PRODNAME}#${IRONAME}" -P $cpp_cmd "${tmp_tll}" $cpp_opt
          else
            log_info "       \"${PRODDIR}/$tll_file\" not found!"
            return 1
          fi
        fi
      fi

    fi   # if check_dependencies

    #-#-#
    log_end_action

  done

}

######################################################################
#
# create_file_packages
#
create_file_packages()
{
#
#	Create the OmniFilePackage profile, and also create a hidden file
#	package for each one defined.
#

[ "$DEBUG" = true ] && set -x

for fp in $FILEPACKS; do
	eval FILEPACKNAME='"$'$fp'"'
	eval FPNAME='"$'$fp'_SHORTNAME"'
	eval FPDEF='"$'$fp'_FPDEF"'
	eval FPCFG='"$'$fp'_SCRIPT"'
	eval FPDIALOG='"$'$fp'_DIALOG"'
	eval FPCONFIG='"$'$fp'_CONFIG_TASKS"'
	eval FPICONDSL='"$'$fp'_ICONDSL"'
	eval FPICONSTATE='"$'$fp'_ICONSTATE"'
	eval FPTARGETS='"$'$fp'_TARGETS"'
	eval FPDEPS='"$'$fp'_DEPS"'

	TARGETS=""
	if [ -n "$FPTARGETS" ]; then
		for target in $FPTARGETS; do
			eval TARG='"$'$target'"'
			TARGETS="$TARGETS \"@ProfileManager:$TARG\""
		done
	fi

	#-#-#
	log_info "checking filepack dependencies"

	if check_dependencies $FPDEPS; then
		#-#-#
		log_start_action "Create File Package $FILEPACKNAME"

		#-#-#
		log_info "Creating state $FPICONSTATE"

		if [ -n "$FPICONDSL" -a ! -f $PRODDIR/$FPICONDSL ]; then
			log_info "Could not find $FPICONDSL ... using default"
		fi
		if [ -n "$FPICONDSL" -a -f $PRODDIR/$FPICONDSL ]; then
			LINK_CO=`wlookup -r Classes SolutionsLink`
			initpres $LINK_CO -class -put \
				state "$FPICONSTATE" $PRODDIR/$FPICONDSL \
					xpm $LINKDIR/NCLink_icon.xpm
		else
			FPICONSTATE=NCLINK
		fi
	
		# Create the file package
		#-#-#
		log_info "Creating Profile Manager $FILEPACKNAME"

		if IsValidProfileManager "$FILEPACKNAME"; then
			#-#-#
			log_info "ProfileManager \"$FILEPACKNAME\" exists!"
		else
			#Create the profile manager
			#-#-#
			log_info "ProfileManager \"$FILEPACKNAME\" does NOT exist, creating"
			wcrtprfmgr @PolicyRegion:"$MEDIUM_PRODNAME#$IRONAME" "$FILEPACKNAME"
		fi
	
		if IsValidFilePackageProfile "$FILEPACKNAME"; then
			#-#-#
			log_info "FilePackage Profile \"$FILEPACKNAME\" exists!"
		else
			#-#-#
			log_info "Creating file package profile \"$FILEPACKNAME\""
			wcrtprf @ProfileManager:"$FILEPACKNAME#$IRONAME" \
				FilePackage "$FILEPACKNAME" 2>&1 > /dev/null
		fi

		#-#-#
		log_success "File Package $FILEPACKNAME creation successful"

		#-#-#
		log_end_action


		if [ "$FPCONFIG" != "TRUE" ]; then
			#-#-#
			log_info "No fp config tasks specified for \"$FILEPACKNAME\"!"
		else
			#-#-#
			log_start_action "Creating OmniFilePackage \"$FILEPACKNAME\""

			eval wcrtsprf $FPICONSTATE \"$MEDIUM_PRODNAME\" OmniFilePackage \
				\"$FILEPACKNAME\" \"$FILEPACKNAME\" $TARGETS

			#-#-#
			log_success "OmniFilePackage \"$FILEPACKNAME\" creation successful"
			#-#-#
			log_end_action
		fi
	fi

done

#
#	Import each file package description.  This is done as a separate step
#	so we don't have to determine the order to create filepackages because
#	of nesting.
#
for fp in $FILEPACKS; do
	eval FILEPACKNAME='"$'$fp'"'
	eval FPNAME='"$'$fp'_SHORTNAME"'
	eval FPDEF='"$'$fp'_FPDEF"'
	eval FPCFG='"$'$fp'_SCRIPT"'
	eval FPDIALOG='"$'$fp'_DIALOG"'
	eval FPDEPS='"$'$fp'_DEPS"'

	#-#-#
	log_info "checking filepack dependencies"

	if check_dependencies $FPDEPS; then
		#-#-#
		log_start_action "Importing definition into File Package $FILEPACKNAME"

		# Import the file package definition
		#-#-#
		log_info "Importing file package definition from $FPDEF"

		wimprtfp -h @ManagedNode:"$MNNAME#$IRONAME" \
			@FilePackage:"$FILEPACKNAME#$IRONAME" < $PRODDIR/$FPDEF

		wsetfpopts -T gen -S $PRODDIR @FilePackage:"$FILEPACKNAME#$IRONAME"

		#-#-#
		log_success "File Package $FILEPACKNAME definition import successful"

		#-#-#
		log_end_action
	fi
done

#
#	Create the configuration tasks
#
if [ "$USE_TLL" = true ]; then
	compile_tll_for_tasks $FILEPACKTASKS
fi

for fp in $FILEPACKTASKS; do
	eval FPTASKDEPS='"$'$fp'_DEPS"'

	if check_dependencies $FPTASKDEPS; then
		eval FPNAME='"$'$fp'"'
		eval FPJOB='"$'$fp'_JOB"'
		eval FPSCRIPT='"$'$fp'_SCRIPT"'
		eval FPUSER='"$'$fp'_USER"'
		eval FPDIALOG='"$'$fp'_DIALOG"'
		eval FPTASKICONDSL='"$'$fp'_ICONDSL"'
		eval FPTASKICONSTATE='"$'$fp'_ICONSTATE"'
		eval FPTASKLIBR='"$'$fp'_LIBR"'

		if [ -z "$FPUSER" ] ; then
			FPUSER=$NULL
		fi

		#-#-#
		log_start_action "Create Configurable File Package $FPNAME"

		if [ -n "$FPDIALOG" -a ! -f $PRODDIR/$FPDIALOG ]; then
			log_info "Could not find $FPDIALOG ... using default"
		fi
		if [ -z "$FPDIALOG" -o ! -f $PRODDIR/$FPDIALOG ] ; then
			FPDIALOG=$NULL
		else
			FPDIALOG=$PRODDIR/$FPDIALOG
		fi

		if [ -n "$FPTASKICONDSL" -a ! -f $PRODDIR/$FPTASKICONDSL ]; then
			log_info "Could not find $FPTASKICONDSL ... using default"
		fi
		if [ -n "$FPTASKICONDSL" -a -f $PRODDIR/$FPTASKICONDSL ]; then
			LINK_CO=`wlookup -r Classes SolutionsLink`
			initpres $LINK_CO -class -put \
				state "$FPTASKICONSTATE" $PRODDIR/$FPTASKICONDSL \
					xpm $LINKDIR/SULink_icon.xpm
		else
			FPTASKICONSTATE=SULINK
		fi
	
		create_task \
			-t "$FPNAME" \
			-m $PRODDIR/$FPSCRIPT \
			-d $FPDIALOG \
			-h "$MNNAME" \
			-r senior \
			-l "$FPTASKLIBR" \
			-u $FPUSER \
			-M parallel \
			-T 60 \
			-j "$FPJOB" \
			-i $FPTASKICONSTATE

		#-#-#
		log_end_action
	fi
done
}

######################################################################
#
# create_sentry_profiles
#
create_sentry_profiles()
{
#
# Create Indicator Collections
#

[ "$DEBUG" = true ] && set -x

for indcoll in $INDCOLLS; do
	eval INDCOLL='"$'$indcoll'"'
	eval R_INDCOLL='"$R_'$indcoll'"'
	eval COLLDEPENDS='"$'$indcoll'_DEPS"'
	eval COLL_ICONDSL='"$'$indcoll'_ICONDSL"'
	eval COLL_ICONSTATE='"$'$indcoll'_ICONSTATE"'

	#-#-#
	log_start_action "Create Indicator collection $INDCOLL"

        if ( [ -z "$COLLDEPENDS" ] && `check_tme_server` ) || \
           ( [ -n "$COLLDEPENDS" ] && `check_dependencies $COLLDEPENDS` ); then
		if IsValidIndicatorCollection "$INDCOLL"; then
			#make sure it gets added:
			SNTCOLL=`wlookup -r IndicatorCollection "$R_INDCOLL"`
			gcoadd "$COLLOBJ" "$SNTCOLL"

			#-#-#
			log_success "$INDCOLL exists"
		else
			SNTCOLL=`wcrtsntcoll "$MEDIUM_PRODNAME#$IRONAME" "$INDCOLL"`
			if [ -n "$COLL_ICONDSL" -a ! -f $PRODDIR/$COLL_ICONDSL ]; then
			  log_info "Could not find $COLL_ICONDSL ... using default"
			fi
			if [ -n "$COLL_ICONDSL" -a -f $PRODDIR/$COLL_ICONDSL ]; then
				wputicon -r IndicatorCollection $COLL_ICONSTATE \
					$PRODDIR/$COLL_ICONDSL \
					$INST_DIR/generic_unix/TME/PLUS/LINK/PM_icon.xpm
				idlcall "$SNTCOLL"  _set_state '"'$COLL_ICONSTATE'"'
			fi
			gcoadd "$COLLOBJ" "$SNTCOLL"
			log_success "$INDCOLL created"
		fi
	fi
	#-#-#
	log_end_action
done
        
#
# Create Sentry Profile Managers
#
for sentrypm in $SENTRY_PMS; do
	eval PMNAME='"$'$sentrypm'"'
	eval R_PMNAME='"$R_'$sentrypm'"'
	eval SPM_INDCOLLS='"$'$sentrypm'_INDCOLLS"'
	eval SPM_DEPENDS='"$'$sentrypm'_DEPS"'
	eval ICONDSL='"$'$sentrypm'_ICONDSL"'
	eval ICONSTATE='"$'$sentrypm'_ICONSTATE"'
	eval PM_TARGETS='"$'$sentrypm'_TARGETS"'

	#-#-#
	log_start_action "Create Sentry Profile Manager $PMNAME"

        if ( [ -z "$SPM_DEPENDS" ] && `check_tme_server` ) || \
           ( [ -n "$SPM_DEPENDS" ] && `check_dependencies $SPM_DEPENDS` ); then
		# Set the subscrition list
		if [ -z "$PM_TARGETS" ]; then
			TARGETS=@ManagedNode:$MNNAME
		else
			TARGETS=""
			for target in $PM_TARGETS; do
				eval TARG='"$'$target'"'
				TARGETS="$TARGETS \"@ProfileManager:$TARG\""
			done
		fi

		if IsValidSentryProfile "$PMNAME"; then
			log_info $R_PMNAME exists, updating subscription list
			eval wpmsub \"$PMNAME\" $TARGETS
		else
			if [ -n "$ICONDSL" -a ! -f $PRODDIR/$ICONDSL ]; then
				log_info "Could not find $ICONDSL ... using default"
			fi
			if [ -n "$ICONDSL" -a -f $PRODDIR/$ICONDSL ]; then
				LINK_CO=`wlookup -r Classes SolutionsLink`
				initpres $LINK_CO -class -put \
					state "$ICONSTATE" $PRODDIR/$ICONDSL \
						xpm $LINKDIR/SLink_icon.xpm
			else
				ICONSTATE=SLINK
			fi

			eval wcrtsprf $ICONSTATE \"$MEDIUM_PRODNAME\" SentryProfile \
				\"$PMNAME\" \
				\"$PMNAME\" \
				$TARGETS
		fi
        
		if IsValidSentryProfile "$PMNAME"; then
			# Profile must run as $root_user
			wsetsntid 0 0 "$R_PMNAME"

			#add precedence levels
			for indcoll in $SPM_INDCOLLS; do
				eval INDCOLL='"$'$indcoll'"'
				eval R_INDCOLL='"$'$indcoll'#$IRONAME"'

				if wsetcoll "$R_INDCOLL" "$R_PMNAME"; then
					log_success "done"
				else
					#-#-#
					log_success "$R_PMNAME in $INDCOLL"
				fi
			done

		        add_cond_levels "$PMNAME" "$TIV_COND_LEVELS"

    		fi
	fi

	#-#-#
	log_end_action
done

#
# Create Sentry Monitors Instances
#
for mon in $MONITORS; do
	eval MONITOR='"$'$mon'"'
	eval MONCOLL='"$'$mon'_MONCAPCOL"'
	eval MONPROG='"$'$mon'_PROG"'
	eval MONARGS='"$'$mon'_ARGS"'
	eval MONSAMP='"$'$mon'_SAMP"'
	eval MONCONDS='"$'$mon'_CONDS"'
	eval MONPMVAR='"$'$mon'_PMVAR"'
	eval MONPMDEPS='"$'$MONPMVAR'_DEPS"'
	eval R_MONPM='"$'$mon'_PM#$IRONAME"'

	# the following rounds seconds to minutes as a workaround for DM 
	# not handling seconds
	if [ "$MONSAMP" ]; then	
	        MONSAMP=`roundoff_rate $MONSAMP`
	fi


        if ( [ -z "$MONPMDEPS" ] && `check_tme_server` ) || \
           ( [ -n "$MONPMDEPS" ] && `check_dependencies $MONPMDEPS` ); then
		log_info "PM Dependencies passed"

		if [ -z "$MONCOLL" ]; then
			MONCOLL=Unix_Sentry
		fi

		#
		# If Sentry monitors are installed, then install the col files
		#
		if $SENTRY_MONITORS_OK ; then
			#-#-#
			log_start_action "Creating Monitoring Collection $MONCOLL"
			if [ -f $PRODDIR/$MONCOLL.col ] ; then
				if mcsl -Ri $PRODDIR/$MONCOLL.col; then
					#-#-#
					if mcsl -Rz "$R_MONPM" > /dev/null 2>&1; then
					  : # update collection on DM3.6 endpts
					fi
					log_success "$MONCOLL installed"    
				else
					#-#-#
					log_failure "error $? installing $MONCOLL"
					return 1
				fi
			else
				log_failure "$MONCOLL.col does not exist!"
				# try using the directory name in case it's a 
				# predefined collection
				MONCOLL=`echo $MONPROG | awk -F/ '{print $1}'`
			fi
			#-#-#
			log_end_action
		else
			log_failure "SENTRY_MONITORS_OK is FALSE!!!!!"
		fi

		WADDMONARGS=""

		#
		#	create a string with the arguments
		#
		for arg in $MONARGS; do
			eval MONARG='"$'$arg'"'
			if [ -n "$MONARG" ] ; then
				WADDMONARGS=' '$WADDMONARGS' -a "'$MONARG'"'
			fi
		done

		#
		#	append the monitoring interval
		#
		if [ -n "$MONSAMP" ] ; then
			WADDMONARGS=' '$WADDMONARGS' -t "'$MONSAMP'"'
		fi

		#
		#	append each severity level and condition clause
		#
		for cond in $MONCONDS; do
			eval CONDLEVEL='"$'$cond'_LEVEL"'
			eval CONDOPER='"$'$cond'_OPER"'
			eval CONDVALUE='"$'$cond'_VALUE"'
			eval CONDCLAUSE='"$'$cond'_TEC_CLAUSE"'
			eval CONDTASK='"$'$cond'_TASK"'
			eval CONDTASKLIB='"$'$cond'_LIBR"'
			eval CONDARGS='"$'$cond'_TASKARGS"'
		        r_libname="${CONDTASKLIB}#${IRONAME}"

			WADDMONTASK=""
			if [ -n "$CONDLEVEL" ] ; then
				WADDMONARGS=' '$WADDMONARGS' -c "'$CONDLEVEL'"'
			fi
			if [ x"$CONDOPER" != x ] ; then
				WADDMONARGS=' '$WADDMONARGS' -R "'$CONDOPER'"'
			fi
			if [ -n "$CONDVALUE" ] ; then
				WADDMONARGS=' '$WADDMONARGS' "'$CONDVALUE'" -i'
			fi
			if [ -n "$CONDCLAUSE" ] ; then
				WADDMONARGS=' '$WADDMONARGS' '$CONDCLAUSE''
			fi
			if [ -n "$CONDTASK" ] ; then
	
				if [ "$USE_TLL" = true ]; then
					# strip blanks, since builder would have
					CONDTASK=`normalize_name "$CONDTASK"`
				fi

				CONDTASKLIB="$r_libname"
				WADDMONTASK=' '$WADDMONTASK' -x "'$CONDTASKLIB'" "'$CONDTASK'"'
				if [ -n "$CONDARGS" ] ; then
					for arg in $CONDARGS; do
						eval CONDARG='"$'$arg'"'
						WADDMONTASK=' '$WADDMONTASK' '"'$CONDARG'"''
					done
				fi
				WADDMONTASK=' '$WADDMONTASK' '"\;"''
				WADDMONARGS=' '$WADDMONARGS' '$WADDMONTASK''
			fi
		done

		#
		#	now try to add the monitor
		#
		#-#-#

		# strip off any leading path info
		MONPROG=`echo $MONPROG | awk -F/ '{print $NF}'`

		# get rid of blanks, since builder would have
		MONITOR=`normalize_name "$MONITOR"`

		log_start_action "Creating monitor $MONPROG"
		if IsValidMonitorCollection $MONCOLL; then
			# Try both MONITOR and MONPROG, predefined's could be 
			#  either!
			if eval waddmon '"$MONCOLL"' '"$MONITOR"' $WADDMONARGS '"$R_MONPM"'; then
				#-#-#
				log_success "$MONCOLL $MONITOR monitor created"
			elif eval waddmon '"$MONCOLL"' '"$MONPROG"' $WADDMONARGS '"$R_MONPM"'; then
				#-#-#
				log_success "$MONCOLL $MONPROG monitor created"
			else
				#-#-#
				log_failure "error $? installing $MONPROG"
				return 1
			fi
		else
			log_failure "$MONCOLL does not exist"
			return 1
		fi
	
	#
	# dependency check failed
	#
	else
		log_info "PM Dependencies failed"
	fi
done
}

######################################################################
#
# set_install_options
#
set_install_options()
{

[ "$DEBUG" = true ] && set -x

#-#-#
log_start_action "Setting install options"

CLO=`wlookup Library`
CO=`idlcall $CLO TMF_SysAdmin::Library::lookup_object \
		'"TMF_imp_TSysAdmin::PolicyDrivenBase" {0}'`

for installopt in $INSTALL_OPTS; do
	eval OPT='"$'$installopt'"'
	eval OPTVAL='"$'$OPT'"'
	eval OPTNAME='"$'$installopt'_NAME"'

	#-#-#
	log_info "Setting $OPT to '$OPTVAL'"

	# create a hidden presentation object to hold the install options
	PRESNAME="$OPTNAME"_"$MNNAME"

	# see if it's already created
	if wlookup -r Presentation "$PRESNAME" > /dev/null 2>& 1; then
		log_info "Presentation Object '$PRESNAME' already exists"
	else
		#-#-#
		log_info "Creating Presentation object '$PRESNAME'"

		MNOID=`objcall 0.0.0 get_host_location`
		PRESOBJ=`idlcall $CO \
					TMF_SysAdmin::InstanceManager::create_instance \
					OBJECT_NIL $MNOID \"$PRESNAME\"`
		initpres "$PRESNAME" -pres-name -create
	fi

	# if the attribute doesn't exist, add it, otherwise set it.
	if wgetattr $OPT @Presentation:"$PRESNAME" > /dev/null 2>& 1; then

		#-#-#
		log_info "Setting attribute $OPT to '$OPTVAL'"

		wsetattr "$OPTVAL" $OPT @Presentation:"$PRESNAME"
	else

		#-#-#
		log_info "Adding attribute $OPT (setting to $OPTVAL)"

		waddattr -v "$OPTVAL" $OPT @Presentation:"$PRESNAME"
	fi
done

log_end_action
}

######################################################################
#
# create_plus_collection
#
create_plus_collection()
{

[ "$DEBUG" = true ] && set -x

#-#-#
log_start_action "Creating top level plus collection"
NULL="__NULL__"

#
# make sure the top level TivoliPlus collections exists on $root_user desktop
#
wcrttopcol

#
# Create the /Plus module within the "TivoliPlus" Collection
#
if [ -z "$PLUSBITMAP" ]; then
  PLUSBITMAP="$PRODDIR/Default.xpm"
fi

if [ -f "$PLUSBITMAP" ]; then
  if [ -f "$PRODDIR/$ICONDSL" ]; then
    wcrtsol "$MEDIUM_PRODNAME" TP$SHORT_PRODNAME "$PLUSBITMAP" $PRODDIR/$ICONDSL
  else
    wcrtsol "$MEDIUM_PRODNAME" TP$SHORT_PRODNAME "$PLUSBITMAP"
  fi
else
  #-#-#
  log_failure "error creating module collection: PLUSBITMAP file $PLUSBITMAP does not exist"
  return 1
fi


#
# Retrieve the OID of the collection just created
#
COLLECTIONGUI=`idlcall "$LIBRARY" lookup_object '"imp_TMF_TGC::CollectionGUI" {0}'`
COLLOBJ=`idlcall "$COLLECTIONGUI" lookup_object '"'$MEDIUM_PRODNAME'" {0}'`
#-#-#
log_end_action
}

######################################################################
#
#	cpp_path
#
cpp_path()
{
  [ "$DEBUG" = true ] && set -x
  if [ ! "$CPP_PATH" ]; then
    if [ x\"$OS\" = x\"Windows_NT\" ] ; then
      CPP_PATH=$BINDIR/tools/cpp.exe
    elif type gcc > /dev/null 2>& 1; then
      GCCLIB=`gcc -print-libgcc-file-name`
      GCCLIBDIR=`dirname ${GCCLIB}`
      CPP_PATH=${GCCLIBDIR}/cpp
    elif type cpp > /dev/null 2>& 1; then
      # extrapolate from our PATH
      CPP_PATH=`type cpp | awk '{print $NF}'`
    else
      CPP_PATH=""
    fi
  fi
  if [ -x "$CPP_PATH" ]; then
    echo "$CPP_PATH"
  else
    echo ""
  fi
}

######################################################################
#
#       is_TMR_server
#
is_TMR_server() {
  ODNUM=`odadmin | grep Dispatcher | awk '{print $3}'`
  if [ $ODNUM -ne 1 ]; then
    return 1
  fi
  return 0
}


######################################################################
#
#       do_configure_TMA
#
do_configure_TMA() {
  [ "$DEBUG" = true ] && set -x
  if is_TMR_server; then
    . $PRODDIR/PLUSconfig-TMA-utils.sh
    TMA_config_full
    TMA_sync_gateways
  fi
}


######################################################################
#
# common_init
#
common_init()
{

[ "$DEBUG" = true ] && set -x

DEFAULT_TASKLIB="$MEDIUM_PRODNAME Utility Task Library"

PATH=$PRODDIR:/bin:/usr/bin:$BINDIR/AEF:$BINDIR/ADE:$BINDIR/tools:$PATH:/usr/ucb:/usr/lib:/usr/ccs/lib
export PATH

#
# Set parallel "R_*" variables with region name appended to avoid collisions
#
for var in $SENTRY_PMS $INDCOLLS $SUBLIST_PMS $TASKLIBS $SENTRY_PM_INDCOLLS $FILEPACKS; do
	eval R_$var='"$'$var'#$IRONAME"'
done

AMS_COND_LEVELS="
  AMS_normal
  AMS_informational
  AMS_warning
  AMS_severe
  AMS_critical
  AMS_fatal
"
TIV_COND_LEVELS="
  normal_zero
  normal
  informational
  warning
  severe
  critical
  fatal
"

#
# cd to the appropriate directory
#
cd $PRODDIR || return 1

#
# Install SolutionsLink Class
#
install_link_class || return 1

#
# Check to see which applications are installed
#
check_apps_installed

}
