#include "misc.h"
#include "apphook.h"
#include "timeutils.h"
#include "timeutils.h"
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#define TEST_ASSERT(x)  \
  do { \
   if (!(x)) \
     { \
       fprintf(stderr, "test assertion failed: " #x " line: %d\n", __LINE__); \
       return 1; \
     } \
  } while (0)

void
set_tz(const char *tz)
{
  char envbuf[32];
  static char *last_setting = NULL;
  
  if (last_setting)
    g_free(last_setting);
  
  snprintf(envbuf, sizeof(envbuf), "TZ=%s", tz);
  last_setting = g_strdup(envbuf);
  putenv(last_setting);
  tzset();
}

void 
testcase(const gchar *zone, time_t utc, long expected_ofs)
{
  long ofs;
  
  clean_time_cache();
  set_tz(zone);
  
  ofs = get_local_timezone_ofs(utc);
  if (ofs != expected_ofs)
    {
      fprintf(stderr, "Timezone offset mismatch: zone: %s, %ld, expected %ld\n", zone, ofs, expected_ofs);
      exit(1);
    }
}

int
test_timezone_2(const time_t stamp_to_test, const char* time_zone)
{
  TimeZoneInfo *info;
  time_t offset, expected_offset;
  
  set_tz(time_zone);
  info = time_zone_info_new(time_zone);
  offset = time_zone_info_get_offset(info, stamp_to_test);
  expected_offset = get_local_timezone_ofs(stamp_to_test);
  if (offset != expected_offset)
    printf("unixtimestamp: %d TimeZoneName (%s) localtime offset(%d), timezone file offset(%d)\n", (gint) stamp_to_test, time_zone, (gint) expected_offset, (gint) offset);
  time_zone_info_free(info);
  return offset == expected_offset;
}

int
test_timezone(const time_t stamp_to_test, const char* time_zone)
{
  return test_timezone_2(stamp_to_test, time_zone) && test_timezone_2(stamp_to_test - 6*30*24*60*60, time_zone) && test_timezone_2(stamp_to_test + 6*30*24*6, time_zone);
}


int 
main(int argc, char *argv[])
{
  gboolean test_all = FALSE;
  
  if (argc > 1)
    test_all = TRUE;
  app_startup();
  /* 2005-10-14 21:47:37 CEST, dst enabled */
  testcase("MET-1METDST", 1129319257, 7200);
  /* 2005-11-14 10:10:00 CET, dst disabled */
  testcase("MET-1METDST", 1131959400, 3600);

  /* 2005-10-14 21:47:37 GMT, no DST */
  testcase("GMT", 1129319257, 0);
  /* 2005-11-14 10:10:00 GMT, no DST */
  testcase("GMT", 1131959400, 0);

  /* 2005-04-03 01:30:00 ESD, DST disabled */
  testcase("EST5EDT", 1112509800, -5*3600);
  /* 2005-04-03 01:59:59 ESD, DST disabled */
  testcase("EST5EDT", 1112511599, -5*3600);
  /* 2005-04-03 03:00:00 EDT, DST enabled */
  testcase("EST5EDT", 1112511600, -4*3600);
  /* 2005-04-03 03:00:01 EDT, DST enabled */
  testcase("EST5EDT", 1112511601, -4*3600);
  /* 2005-10-30 01:59:59 EDT, DST enabled */
  testcase("EST5EDT", 1130651999, -4*3600);
  /* 2005-10-30 01:00:00 EST, DST disabled */
  testcase("EST5EDT", 1130652000, -5*3600);

#ifdef __linux__
  
  /* NOTE: not all of our build systems have been upgraded to work correctly
   * with post 2007 years. Therefore we restrict their tests to Linux which
   * work ok. */
  
  /* USA DST change in 2007, 2nd sunday of March instead of 1st Sunday of April */
  /* 2007-03-11 01:00:00 EST, DST disabled */
  testcase("EST5EDT", 1173592800, -5*3600);
  /* 2007-03-11 01:59:59 EST, DST disabled */
  testcase("EST5EDT", 1173596399, -5*3600);
  /* 2007-03-11 03:00:00 EST, DST enabled */
  testcase("EST5EDT", 1173596400, -4*3600);
  /* 2007-11-04 01:59:59 EST, DST enabled */
  testcase("EST5EDT", 1194155999, -4*3600);
  /* 2007-11-04 01:00:00 EST, DST disabled */
  testcase("EST5EDT", 1194156000, -5*3600);
#endif

#ifdef __linux__  
  if (test_all)
    {

      /* Oct 31 01:59:59 2004 (EST) +1000 */
      testcase("Australia/Victoria", 1099151999, 10*3600);
      /* Oct 31 03:00:00 2004 (EST) +1100 */
      testcase("Australia/Victoria", 1099152000, 11*3600);
      /* Mar 27 02:59:59 2005 (EST) +1100 */
      testcase("Australia/Victoria", 1111852799, 11*3600);
      /* Mar 27 02:00:00 2005 (EST) +1000 */
      testcase("Australia/Victoria", 1111852800, 10*3600);

      /* Oct  2 01:59:59 2004 (NZT) +1200 */
      testcase("NZ", 1128175199, 12*3600);
      /* Oct  2 03:00:00 2004 (NZDT) +1300 */
      testcase("NZ", 1128175200, 13*3600);
      /* Mar 20 02:59:59 2005 (NZDT) +1300 */
      testcase("NZ", 1111240799, 13*3600);
      /* Mar 20 02:00:00 2005 (NZT) +1200 */
      testcase("NZ", 1111240800, 12*3600);


      TEST_ASSERT(test_timezone(time(NULL), "America/Louisville"));
      TEST_ASSERT(test_timezone(time(NULL), "Hongkong"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Budapest"));
      TEST_ASSERT(test_timezone(1288486800, "Europe/Budapest"));
      TEST_ASSERT(test_timezone(1288486860, "Europe/Budapest"));
      TEST_ASSERT(test_timezone(1288486740, "Europe/Budapest"));
      TEST_ASSERT(test_timezone(1288486800-1800, "Europe/Budapest"));
      TEST_ASSERT(test_timezone(1288486800+1800, "Europe/Budapest"));
      TEST_ASSERT(test_timezone(1288486800-3600, "Europe/Budapest"));
      TEST_ASSERT(test_timezone(1288486800+3600, "Europe/Budapest"));
      TEST_ASSERT(test_timezone(1288486800-3601, "Europe/Budapest"));
      TEST_ASSERT(test_timezone(1288486800+3601, "Europe/Budapest"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Kuala_Lumpur"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Japan"));
      TEST_ASSERT(test_timezone(time(NULL), "CST6CDT"));
      TEST_ASSERT(test_timezone(time(NULL), "US/Pacific"));
      TEST_ASSERT(test_timezone(time(NULL), "US/Indiana-Starke"));
      TEST_ASSERT(test_timezone(time(NULL), "US/Samoa"));
      TEST_ASSERT(test_timezone(time(NULL), "US/Arizona"));
      TEST_ASSERT(test_timezone(time(NULL), "US/Aleutian"));
      TEST_ASSERT(test_timezone(time(NULL), "US/Michigan"));
      TEST_ASSERT(test_timezone(time(NULL), "US/Alaska"));
      TEST_ASSERT(test_timezone(time(NULL), "US/Central"));
      TEST_ASSERT(test_timezone(time(NULL), "US/Hawaii"));
      TEST_ASSERT(test_timezone(time(NULL), "US/East-Indiana"));
      TEST_ASSERT(test_timezone(time(NULL), "US/Eastern"));
      TEST_ASSERT(test_timezone(time(NULL), "US/Mountain"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Noumea"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Samoa"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Apia"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Auckland"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Fakaofo"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Saipan"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Easter"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Norfolk"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Kosrae"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Tarawa"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Tahiti"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Pago_Pago"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Majuro"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Guam"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Ponape"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Tongatapu"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Funafuti"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Wallis"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Efate"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Marquesas"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Enderbury"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Pitcairn"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Yap"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Wake"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Johnston"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Kwajalein"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Chatham"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Gambier"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Galapagos"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Kiritimati"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Honolulu"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Truk"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Midway"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Fiji"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Rarotonga"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Guadalcanal"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Nauru"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Palau"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Port_Moresby"));
      TEST_ASSERT(test_timezone(time(NULL), "Pacific/Niue"));
      TEST_ASSERT(test_timezone(time(NULL), "GMT"));
      TEST_ASSERT(test_timezone(time(NULL), "Hongkong"));
      TEST_ASSERT(test_timezone(time(NULL), "ROK"));
      TEST_ASSERT(test_timezone(time(NULL), "Navajo"));
      TEST_ASSERT(test_timezone(time(NULL), "ROC"));
      TEST_ASSERT(test_timezone(time(NULL), "WET"));
      TEST_ASSERT(test_timezone(time(NULL), "posixrules"));
      TEST_ASSERT(test_timezone(time(NULL), "CET"));
      TEST_ASSERT(test_timezone(time(NULL), "MET"));
      TEST_ASSERT(test_timezone(time(NULL), "MST"));
      TEST_ASSERT(test_timezone(time(NULL), "Turkey"));
      TEST_ASSERT(test_timezone(time(NULL), "Zulu"));
      TEST_ASSERT(test_timezone(time(NULL), "GMT+0"));
      TEST_ASSERT(test_timezone(time(NULL), "Canada/Saskatchewan"));
      TEST_ASSERT(test_timezone(time(NULL), "Canada/Pacific"));
      TEST_ASSERT(test_timezone(time(NULL), "Canada/Yukon"));
      TEST_ASSERT(test_timezone(time(NULL), "Canada/East-Saskatchewan"));
      TEST_ASSERT(test_timezone(time(NULL), "Canada/Newfoundland"));
      TEST_ASSERT(test_timezone(time(NULL), "Canada/Central"));
      TEST_ASSERT(test_timezone(time(NULL), "Canada/Eastern"));
      TEST_ASSERT(test_timezone(time(NULL), "Canada/Atlantic"));
      TEST_ASSERT(test_timezone(time(NULL), "Canada/Mountain"));
      TEST_ASSERT(test_timezone(time(NULL), "Singapore"));
      TEST_ASSERT(test_timezone(time(NULL), "UCT"));
      TEST_ASSERT(test_timezone(time(NULL), "Poland"));
      TEST_ASSERT(test_timezone(time(NULL), "Chile/Continental"));
      TEST_ASSERT(test_timezone(time(NULL), "Chile/EasterIsland"));
      TEST_ASSERT(test_timezone(time(NULL), "Portugal"));
      TEST_ASSERT(test_timezone(time(NULL), "Egypt"));
      TEST_ASSERT(test_timezone(time(NULL), "Japan"));
      TEST_ASSERT(test_timezone(time(NULL), "Iran"));
      TEST_ASSERT(test_timezone(time(NULL), "EET"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Oslo"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Bratislava"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Gibraltar"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Skopje"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Simferopol"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Zurich"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Vienna"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Paris"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Jersey"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Tallinn"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Madrid"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Volgograd"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Zaporozhye"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Mariehamn"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Vaduz"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Moscow"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Stockholm"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Minsk"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Andorra"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Istanbul"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Tiraspol"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Podgorica"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Bucharest"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Ljubljana"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Brussels"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Amsterdam"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Riga"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Tirane"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Berlin"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Guernsey"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Warsaw"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Vatican"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Malta"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Nicosia"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Budapest"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Kaliningrad"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Sarajevo"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Isle_of_Man"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Rome"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/London"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Vilnius"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Samara"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Belfast"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Athens"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Copenhagen"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Belgrade"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Sofia"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/San_Marino"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Helsinki"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Uzhgorod"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Monaco"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Prague"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Zagreb"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Lisbon"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Chisinau"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Dublin"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Luxembourg"));
      TEST_ASSERT(test_timezone(time(NULL), "Europe/Kiev"));
      TEST_ASSERT(test_timezone(time(NULL), "Jamaica"));
      TEST_ASSERT(test_timezone(time(NULL), "Indian/Chagos"));
      TEST_ASSERT(test_timezone(time(NULL), "Indian/Comoro"));
      TEST_ASSERT(test_timezone(time(NULL), "Indian/Mauritius"));
      TEST_ASSERT(test_timezone(time(NULL), "Indian/Mayotte"));
      TEST_ASSERT(test_timezone(time(NULL), "Indian/Kerguelen"));
      TEST_ASSERT(test_timezone(time(NULL), "Indian/Maldives"));
      TEST_ASSERT(test_timezone(time(NULL), "Indian/Antananarivo"));
      TEST_ASSERT(test_timezone(time(NULL), "Indian/Mahe"));
      TEST_ASSERT(test_timezone(time(NULL), "Indian/Cocos"));
      TEST_ASSERT(test_timezone(time(NULL), "Indian/Christmas"));
      TEST_ASSERT(test_timezone(time(NULL), "Indian/Reunion"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/CST6CDT"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/US/Pacific"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/US/Indiana-Starke"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/US/Samoa"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/US/Arizona"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/US/Aleutian"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/US/Michigan"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/US/Alaska"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/US/Central"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/US/Hawaii"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/US/East-Indiana"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/US/Eastern"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/US/Mountain"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Noumea"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Samoa"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Apia"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Auckland"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Fakaofo"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Saipan"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Easter"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Norfolk"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Kosrae"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Tarawa"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Tahiti"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Pago_Pago"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Majuro"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Guam"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Ponape"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Tongatapu"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Funafuti"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Wallis"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Efate"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Marquesas"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Enderbury"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Pitcairn"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Yap"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Wake"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Johnston"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Kwajalein"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Chatham"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Gambier"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Galapagos"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Kiritimati"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Honolulu"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Truk"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Midway"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Fiji"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Rarotonga"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Guadalcanal"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Nauru"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Palau"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Port_Moresby"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Pacific/Niue"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/GMT"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Hongkong"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/ROK"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Navajo"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/ROC"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/WET"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/CET"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/MET"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/MST"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Turkey"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Zulu"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/GMT+0"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Canada/Saskatchewan"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Canada/Pacific"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Canada/Yukon"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Canada/East-Saskatchewan"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Canada/Newfoundland"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Canada/Central"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Canada/Eastern"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Canada/Atlantic"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Canada/Mountain"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Singapore"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/UCT"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Poland"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Chile/Continental"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Chile/EasterIsland"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Portugal"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Egypt"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Japan"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Iran"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/EET"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Oslo"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Bratislava"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Gibraltar"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Skopje"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Simferopol"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Zurich"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Vienna"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Paris"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Jersey"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Tallinn"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Madrid"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Volgograd"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Zaporozhye"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Mariehamn"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Vaduz"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Moscow"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Stockholm"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Minsk"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Andorra"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Istanbul"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Tiraspol"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Podgorica"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Bucharest"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Ljubljana"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Brussels"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Amsterdam"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Riga"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Tirane"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Berlin"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Guernsey"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Warsaw"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Vatican"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Malta"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Nicosia"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Budapest"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Kaliningrad"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Sarajevo"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Isle_of_Man"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Rome"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/London"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Vilnius"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Samara"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Belfast"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Athens"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Copenhagen"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Belgrade"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Sofia"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/San_Marino"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Helsinki"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Uzhgorod"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Monaco"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Prague"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Zagreb"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Lisbon"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Chisinau"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Dublin"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Luxembourg"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Europe/Kiev"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Jamaica"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Indian/Chagos"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Indian/Comoro"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Indian/Mauritius"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Indian/Mayotte"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Indian/Kerguelen"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Indian/Maldives"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Indian/Antananarivo"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Indian/Mahe"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Indian/Cocos"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Indian/Christmas"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Indian/Reunion"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Accra"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Lubumbashi"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Bangui"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Asmara"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Freetown"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Mbabane"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Djibouti"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Ndjamena"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Porto-Novo"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Nouakchott"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Brazzaville"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Tunis"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Dar_es_Salaam"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Lusaka"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Banjul"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Sao_Tome"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Monrovia"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Lagos"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Conakry"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Tripoli"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Algiers"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Casablanca"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Lome"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Bamako"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Nairobi"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Douala"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Addis_Ababa"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/El_Aaiun"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Ceuta"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Harare"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Libreville"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Johannesburg"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Timbuktu"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Niamey"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Windhoek"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Bissau"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Maputo"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Kigali"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Dakar"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Ouagadougou"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Gaborone"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Khartoum"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Bujumbura"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Luanda"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Malabo"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Asmera"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Maseru"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Abidjan"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Kinshasa"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Blantyre"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Cairo"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Kampala"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Africa/Mogadishu"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Universal"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/EST"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Greenwich"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Eire"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Qatar"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Makassar"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Colombo"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Chungking"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Macau"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Choibalsan"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Rangoon"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Harbin"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Yerevan"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Brunei"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Omsk"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Chongqing"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Tbilisi"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Tehran"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Manila"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Yakutsk"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Qyzylorda"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Dhaka"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Singapore"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Jakarta"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Novosibirsk"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Saigon"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Krasnoyarsk"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Kabul"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Bahrain"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Urumqi"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Thimbu"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Ulan_Bator"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Taipei"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Tashkent"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Dacca"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Aqtau"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Seoul"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Istanbul"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Tel_Aviv"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Almaty"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Phnom_Penh"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Baku"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Anadyr"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Aqtobe"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Kuwait"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Irkutsk"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Ulaanbaatar"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Tokyo"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Gaza"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Riyadh87"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Yekaterinburg"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Riyadh88"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Kolkata"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Nicosia"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Jayapura"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Shanghai"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Pyongyang"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Macao"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Dushanbe"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Kuching"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Vientiane"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Riyadh"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Dili"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Samarkand"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Ashkhabad"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Calcutta"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Hong_Kong"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Sakhalin"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Beirut"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Damascus"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Katmandu"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Jerusalem"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Riyadh89"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Vladivostok"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Kamchatka"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Dubai"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Kashgar"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Ashgabat"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Amman"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Karachi"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Bangkok"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Oral"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Thimphu"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Bishkek"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Baghdad"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Kuala_Lumpur"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Pontianak"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Ho_Chi_Minh"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Ujung_Pandang"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Muscat"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Aden"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Hovd"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Asia/Magadan"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/EST5EDT"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/PST8PDT"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Atlantic/Bermuda"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Atlantic/St_Helena"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Atlantic/Cape_Verde"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Atlantic/Stanley"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Atlantic/Azores"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Atlantic/Jan_Mayen"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Atlantic/Reykjavik"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Atlantic/Madeira"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Atlantic/Canary"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Atlantic/Faeroe"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Atlantic/Faroe"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Atlantic/South_Georgia"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Kwajalein"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/UTC"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/GMT-0"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/MST7MDT"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/GB-Eire"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/PRC"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/SystemV/CST6CDT"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/SystemV/CST6"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/SystemV/PST8"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/SystemV/YST9"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/SystemV/EST5"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/SystemV/AST4"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/SystemV/AST4ADT"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/SystemV/YST9YDT"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/SystemV/EST5EDT"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/SystemV/PST8PDT"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/SystemV/MST7MDT"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/SystemV/HST10"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/SystemV/MST7"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Arctic/Longyearbyen"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Cuba"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Israel"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT-3"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT+1"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT-5"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT-13"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT-1"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT-9"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/Zulu"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT+0"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/UCT"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT+12"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT+9"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT-6"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/Universal"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT-2"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/Greenwich"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT+3"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT+8"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/UTC"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT-0"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT-14"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT+10"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT+4"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT+5"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT-12"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT-8"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT+7"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT-11"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT+6"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT0"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT-7"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT+11"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT-4"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT+2"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Etc/GMT-10"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/HST"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Iceland"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Mexico/BajaNorte"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Mexico/BajaSur"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Mexico/General"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Mideast/Riyadh87"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Mideast/Riyadh88"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Mideast/Riyadh89"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Antarctica/Davis"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Antarctica/DumontDUrville"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Antarctica/Syowa"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Antarctica/Palmer"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Antarctica/Casey"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Antarctica/Rothera"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Antarctica/Mawson"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Antarctica/McMurdo"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Antarctica/South_Pole"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Antarctica/Vostok"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Curacao"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/St_Lucia"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Managua"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Lima"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Nipigon"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Cayenne"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Detroit"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Kentucky/Louisville"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Kentucky/Monticello"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Belem"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Jujuy"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Godthab"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Guatemala"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Atka"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Montreal"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Resolute"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Thunder_Bay"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/North_Dakota/New_Salem"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/North_Dakota/Center"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Panama"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Los_Angeles"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Whitehorse"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Santiago"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Iqaluit"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Dawson"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Juneau"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Thule"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Fortaleza"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Montevideo"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Tegucigalpa"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Port-au-Prince"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Guadeloupe"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Coral_Harbour"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Monterrey"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Anguilla"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Antigua"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Campo_Grande"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Buenos_Aires"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Maceio"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/New_York"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Puerto_Rico"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Noronha"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Sao_Paulo"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Cancun"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Aruba"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Yellowknife"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Knox_IN"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Halifax"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Grand_Turk"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Vancouver"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Bogota"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Santo_Domingo"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Tortola"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Blanc-Sablon"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/St_Thomas"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Scoresbysund"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Jamaica"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/El_Salvador"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Montserrat"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Martinique"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Nassau"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Indianapolis"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Pangnirtung"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Port_of_Spain"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Mexico_City"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Denver"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Dominica"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Eirunepe"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Atikokan"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Glace_Bay"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Rainy_River"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/St_Barthelemy"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Miquelon"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Indiana/Vevay"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Indiana/Vincennes"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Indiana/Marengo"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Indiana/Petersburg"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Indiana/Tell_City"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Indiana/Knox"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Indiana/Indianapolis"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Indiana/Winamac"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Menominee"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Porto_Velho"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Phoenix"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Argentina/San_Juan"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Argentina/Jujuy"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Argentina/Ushuaia"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Argentina/Buenos_Aires"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Argentina/La_Rioja"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Argentina/ComodRivadavia"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Argentina/Tucuman"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Argentina/San_Luis"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Argentina/Rio_Gallegos"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Argentina/Mendoza"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Argentina/Cordoba"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Argentina/Catamarca"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Dawson_Creek"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Merida"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Moncton"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Goose_Bay"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Grenada"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Barbados"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Tijuana"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Regina"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Ensenada"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Louisville"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Edmonton"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Bahia"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Nome"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Guayaquil"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/La_Paz"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Costa_Rica"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Mazatlan"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Havana"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Marigot"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Mendoza"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Virgin"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Manaus"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Rosario"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Boa_Vista"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Winnipeg"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Paramaribo"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Danmarkshavn"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Caracas"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Swift_Current"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/St_Johns"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Araguaina"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Adak"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Cordoba"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Fort_Wayne"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Catamarca"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Recife"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Toronto"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Anchorage"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/St_Vincent"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/St_Kitts"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Chihuahua"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Cayman"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Belize"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Cambridge_Bay"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Cuiaba"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Chicago"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Guyana"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Inuvik"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Asuncion"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Porto_Acre"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Hermosillo"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Shiprock"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Rio_Branco"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Yakutat"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Rankin_Inlet"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/America/Boise"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Brazil/West"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Brazil/Acre"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Brazil/East"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Brazil/DeNoronha"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/GMT0"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Libya"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/W-SU"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/NZ-CHAT"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Factory"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/GB"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Australia/West"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Australia/Canberra"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Australia/Broken_Hill"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Australia/Eucla"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Australia/Currie"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Australia/South"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Australia/Lindeman"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Australia/Hobart"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Australia/Perth"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Australia/Yancowinna"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Australia/Victoria"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Australia/Sydney"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Australia/North"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Australia/Adelaide"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Australia/ACT"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Australia/Melbourne"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Australia/Tasmania"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Australia/Darwin"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Australia/Brisbane"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Australia/LHI"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Australia/NSW"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Australia/Queensland"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/Australia/Lord_Howe"));
      TEST_ASSERT(test_timezone(time(NULL), "posix/NZ"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Accra"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Lubumbashi"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Bangui"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Asmara"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Freetown"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Mbabane"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Djibouti"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Ndjamena"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Porto-Novo"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Nouakchott"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Brazzaville"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Tunis"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Dar_es_Salaam"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Lusaka"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Banjul"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Sao_Tome"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Monrovia"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Lagos"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Conakry"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Tripoli"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Algiers"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Casablanca"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Lome"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Bamako"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Nairobi"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Douala"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Addis_Ababa"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/El_Aaiun"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Ceuta"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Harare"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Libreville"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Johannesburg"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Timbuktu"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Niamey"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Windhoek"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Bissau"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Maputo"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Kigali"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Dakar"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Ouagadougou"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Gaborone"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Khartoum"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Bujumbura"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Luanda"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Malabo"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Asmera"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Maseru"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Abidjan"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Kinshasa"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Blantyre"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Cairo"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Kampala"));
      TEST_ASSERT(test_timezone(time(NULL), "Africa/Mogadishu"));
      TEST_ASSERT(test_timezone(time(NULL), "Universal"));
      TEST_ASSERT(test_timezone(time(NULL), "EST"));
      TEST_ASSERT(test_timezone(time(NULL), "right/CST6CDT"));
      TEST_ASSERT(test_timezone(time(NULL), "right/US/Pacific"));
      TEST_ASSERT(test_timezone(time(NULL), "right/US/Indiana-Starke"));
      TEST_ASSERT(test_timezone(time(NULL), "right/US/Samoa"));
      TEST_ASSERT(test_timezone(time(NULL), "right/US/Arizona"));
      TEST_ASSERT(test_timezone(time(NULL), "right/US/Aleutian"));
      TEST_ASSERT(test_timezone(time(NULL), "right/US/Michigan"));
      TEST_ASSERT(test_timezone(time(NULL), "right/US/Alaska"));
      TEST_ASSERT(test_timezone(time(NULL), "right/US/Central"));
      TEST_ASSERT(test_timezone(time(NULL), "right/US/Hawaii"));
      TEST_ASSERT(test_timezone(time(NULL), "right/US/East-Indiana"));
      TEST_ASSERT(test_timezone(time(NULL), "right/US/Eastern"));
      TEST_ASSERT(test_timezone(time(NULL), "right/US/Mountain"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Noumea"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Samoa"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Apia"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Auckland"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Fakaofo"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Saipan"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Easter"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Norfolk"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Kosrae"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Tarawa"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Tahiti"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Pago_Pago"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Majuro"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Guam"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Ponape"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Tongatapu"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Funafuti"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Wallis"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Efate"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Marquesas"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Enderbury"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Pitcairn"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Yap"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Wake"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Johnston"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Kwajalein"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Chatham"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Gambier"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Galapagos"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Kiritimati"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Honolulu"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Truk"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Midway"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Fiji"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Rarotonga"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Guadalcanal"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Nauru"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Palau"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Port_Moresby"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Pacific/Niue"));
      TEST_ASSERT(test_timezone(time(NULL), "right/GMT"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Hongkong"));
      TEST_ASSERT(test_timezone(time(NULL), "right/ROK"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Navajo"));
      TEST_ASSERT(test_timezone(time(NULL), "right/ROC"));
      TEST_ASSERT(test_timezone(time(NULL), "right/WET"));
      TEST_ASSERT(test_timezone(time(NULL), "right/CET"));
      TEST_ASSERT(test_timezone(time(NULL), "right/MET"));
      TEST_ASSERT(test_timezone(time(NULL), "right/MST"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Turkey"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Zulu"));
      TEST_ASSERT(test_timezone(time(NULL), "right/GMT+0"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Canada/Saskatchewan"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Canada/Pacific"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Canada/Yukon"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Canada/East-Saskatchewan"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Canada/Newfoundland"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Canada/Central"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Canada/Eastern"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Canada/Atlantic"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Canada/Mountain"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Singapore"));
      TEST_ASSERT(test_timezone(time(NULL), "right/UCT"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Poland"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Chile/Continental"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Chile/EasterIsland"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Portugal"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Egypt"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Japan"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Iran"));
      TEST_ASSERT(test_timezone(time(NULL), "right/EET"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Oslo"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Bratislava"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Gibraltar"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Skopje"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Simferopol"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Zurich"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Vienna"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Paris"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Jersey"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Tallinn"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Madrid"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Volgograd"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Zaporozhye"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Mariehamn"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Vaduz"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Moscow"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Stockholm"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Minsk"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Andorra"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Istanbul"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Tiraspol"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Podgorica"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Bucharest"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Ljubljana"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Brussels"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Amsterdam"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Riga"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Tirane"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Berlin"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Guernsey"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Warsaw"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Vatican"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Malta"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Nicosia"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Budapest"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Kaliningrad"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Sarajevo"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Isle_of_Man"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Rome"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/London"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Vilnius"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Samara"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Belfast"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Athens"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Copenhagen"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Belgrade"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Sofia"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/San_Marino"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Helsinki"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Uzhgorod"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Monaco"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Prague"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Zagreb"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Lisbon"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Chisinau"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Dublin"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Luxembourg"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Europe/Kiev"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Jamaica"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Indian/Chagos"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Indian/Comoro"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Indian/Mauritius"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Indian/Mayotte"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Indian/Kerguelen"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Indian/Maldives"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Indian/Antananarivo"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Indian/Mahe"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Indian/Cocos"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Indian/Christmas"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Indian/Reunion"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Accra"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Lubumbashi"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Bangui"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Asmara"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Freetown"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Mbabane"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Djibouti"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Ndjamena"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Porto-Novo"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Nouakchott"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Brazzaville"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Tunis"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Dar_es_Salaam"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Lusaka"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Banjul"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Sao_Tome"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Monrovia"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Lagos"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Conakry"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Tripoli"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Algiers"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Casablanca"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Lome"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Bamako"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Nairobi"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Douala"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Addis_Ababa"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/El_Aaiun"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Ceuta"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Harare"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Libreville"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Johannesburg"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Timbuktu"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Niamey"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Windhoek"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Bissau"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Maputo"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Kigali"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Dakar"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Ouagadougou"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Gaborone"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Khartoum"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Bujumbura"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Luanda"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Malabo"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Asmera"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Maseru"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Abidjan"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Kinshasa"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Blantyre"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Cairo"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Kampala"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Africa/Mogadishu"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Universal"));
      TEST_ASSERT(test_timezone(time(NULL), "right/EST"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Greenwich"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Eire"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Qatar"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Makassar"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Colombo"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Chungking"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Macau"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Choibalsan"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Rangoon"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Harbin"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Yerevan"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Brunei"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Omsk"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Chongqing"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Tbilisi"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Tehran"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Manila"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Yakutsk"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Qyzylorda"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Dhaka"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Singapore"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Jakarta"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Novosibirsk"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Saigon"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Krasnoyarsk"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Kabul"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Bahrain"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Urumqi"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Thimbu"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Ulan_Bator"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Taipei"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Tashkent"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Dacca"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Aqtau"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Seoul"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Istanbul"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Tel_Aviv"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Almaty"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Phnom_Penh"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Baku"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Anadyr"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Aqtobe"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Kuwait"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Irkutsk"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Ulaanbaatar"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Tokyo"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Gaza"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Riyadh87"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Yekaterinburg"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Riyadh88"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Kolkata"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Nicosia"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Jayapura"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Shanghai"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Pyongyang"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Macao"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Dushanbe"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Kuching"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Vientiane"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Riyadh"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Dili"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Samarkand"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Ashkhabad"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Calcutta"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Hong_Kong"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Sakhalin"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Beirut"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Damascus"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Katmandu"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Jerusalem"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Riyadh89"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Vladivostok"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Kamchatka"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Dubai"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Kashgar"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Ashgabat"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Amman"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Karachi"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Bangkok"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Oral"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Thimphu"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Bishkek"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Baghdad"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Kuala_Lumpur"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Pontianak"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Ho_Chi_Minh"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Ujung_Pandang"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Muscat"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Aden"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Hovd"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Asia/Magadan"));
      TEST_ASSERT(test_timezone(time(NULL), "right/EST5EDT"));
      TEST_ASSERT(test_timezone(time(NULL), "right/PST8PDT"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Atlantic/Bermuda"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Atlantic/St_Helena"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Atlantic/Cape_Verde"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Atlantic/Stanley"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Atlantic/Azores"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Atlantic/Jan_Mayen"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Atlantic/Reykjavik"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Atlantic/Madeira"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Atlantic/Canary"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Atlantic/Faeroe"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Atlantic/Faroe"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Atlantic/South_Georgia"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Kwajalein"));
      TEST_ASSERT(test_timezone(time(NULL), "right/UTC"));
      TEST_ASSERT(test_timezone(time(NULL), "right/GMT-0"));
      TEST_ASSERT(test_timezone(time(NULL), "right/MST7MDT"));
      TEST_ASSERT(test_timezone(time(NULL), "right/GB-Eire"));
      TEST_ASSERT(test_timezone(time(NULL), "right/PRC"));
      TEST_ASSERT(test_timezone(time(NULL), "right/SystemV/CST6CDT"));
      TEST_ASSERT(test_timezone(time(NULL), "right/SystemV/CST6"));
      TEST_ASSERT(test_timezone(time(NULL), "right/SystemV/PST8"));
      TEST_ASSERT(test_timezone(time(NULL), "right/SystemV/YST9"));
      TEST_ASSERT(test_timezone(time(NULL), "right/SystemV/EST5"));
      TEST_ASSERT(test_timezone(time(NULL), "right/SystemV/AST4"));
      TEST_ASSERT(test_timezone(time(NULL), "right/SystemV/AST4ADT"));
      TEST_ASSERT(test_timezone(time(NULL), "right/SystemV/YST9YDT"));
      TEST_ASSERT(test_timezone(time(NULL), "right/SystemV/EST5EDT"));
      TEST_ASSERT(test_timezone(time(NULL), "right/SystemV/PST8PDT"));
      TEST_ASSERT(test_timezone(time(NULL), "right/SystemV/MST7MDT"));
      TEST_ASSERT(test_timezone(time(NULL), "right/SystemV/HST10"));
      TEST_ASSERT(test_timezone(time(NULL), "right/SystemV/MST7"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Arctic/Longyearbyen"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Cuba"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Israel"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT-3"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT+1"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT-5"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT-13"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT-1"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT-9"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/Zulu"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT+0"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/UCT"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT+12"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT+9"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT-6"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/Universal"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT-2"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/Greenwich"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT+3"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT+8"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/UTC"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT-0"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT-14"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT+10"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT+4"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT+5"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT-12"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT-8"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT+7"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT-11"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT+6"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT0"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT-7"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT+11"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT-4"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT+2"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Etc/GMT-10"));
      TEST_ASSERT(test_timezone(time(NULL), "right/HST"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Iceland"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Mexico/BajaNorte"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Mexico/BajaSur"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Mexico/General"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Mideast/Riyadh87"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Mideast/Riyadh88"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Mideast/Riyadh89"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Antarctica/Davis"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Antarctica/DumontDUrville"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Antarctica/Syowa"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Antarctica/Palmer"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Antarctica/Casey"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Antarctica/Rothera"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Antarctica/Mawson"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Antarctica/McMurdo"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Antarctica/South_Pole"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Antarctica/Vostok"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Curacao"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/St_Lucia"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Managua"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Lima"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Nipigon"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Cayenne"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Detroit"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Kentucky/Louisville"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Kentucky/Monticello"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Belem"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Jujuy"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Godthab"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Guatemala"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Atka"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Montreal"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Resolute"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Thunder_Bay"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/North_Dakota/New_Salem"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/North_Dakota/Center"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Panama"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Los_Angeles"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Whitehorse"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Santiago"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Iqaluit"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Dawson"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Juneau"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Thule"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Fortaleza"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Montevideo"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Tegucigalpa"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Port-au-Prince"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Guadeloupe"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Coral_Harbour"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Monterrey"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Anguilla"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Antigua"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Campo_Grande"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Buenos_Aires"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Maceio"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/New_York"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Puerto_Rico"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Noronha"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Sao_Paulo"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Cancun"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Aruba"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Yellowknife"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Knox_IN"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Halifax"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Grand_Turk"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Vancouver"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Bogota"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Santo_Domingo"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Tortola"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Blanc-Sablon"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/St_Thomas"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Scoresbysund"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Jamaica"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/El_Salvador"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Montserrat"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Martinique"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Nassau"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Indianapolis"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Pangnirtung"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Port_of_Spain"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Mexico_City"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Denver"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Dominica"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Eirunepe"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Atikokan"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Glace_Bay"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Rainy_River"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/St_Barthelemy"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Miquelon"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Indiana/Vevay"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Indiana/Vincennes"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Indiana/Marengo"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Indiana/Petersburg"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Indiana/Tell_City"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Indiana/Knox"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Indiana/Indianapolis"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Indiana/Winamac"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Menominee"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Porto_Velho"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Phoenix"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Argentina/San_Juan"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Argentina/Jujuy"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Argentina/Ushuaia"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Argentina/Buenos_Aires"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Argentina/La_Rioja"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Argentina/ComodRivadavia"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Argentina/Tucuman"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Argentina/San_Luis"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Argentina/Rio_Gallegos"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Argentina/Mendoza"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Argentina/Cordoba"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Argentina/Catamarca"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Dawson_Creek"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Merida"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Moncton"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Goose_Bay"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Grenada"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Barbados"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Tijuana"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Regina"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Ensenada"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Louisville"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Edmonton"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Bahia"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Nome"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Guayaquil"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/La_Paz"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Costa_Rica"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Mazatlan"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Havana"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Marigot"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Mendoza"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Virgin"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Manaus"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Rosario"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Boa_Vista"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Winnipeg"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Paramaribo"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Danmarkshavn"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Caracas"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Swift_Current"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/St_Johns"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Araguaina"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Adak"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Cordoba"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Fort_Wayne"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Catamarca"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Recife"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Toronto"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Anchorage"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/St_Vincent"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/St_Kitts"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Chihuahua"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Cayman"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Belize"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Cambridge_Bay"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Cuiaba"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Chicago"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Guyana"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Inuvik"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Asuncion"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Porto_Acre"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Hermosillo"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Shiprock"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Rio_Branco"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Yakutat"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Rankin_Inlet"));
      TEST_ASSERT(test_timezone(time(NULL), "right/America/Boise"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Brazil/West"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Brazil/Acre"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Brazil/East"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Brazil/DeNoronha"));
      TEST_ASSERT(test_timezone(time(NULL), "right/GMT0"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Libya"));
      TEST_ASSERT(test_timezone(time(NULL), "right/W-SU"));
      TEST_ASSERT(test_timezone(time(NULL), "right/NZ-CHAT"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Factory"));
      TEST_ASSERT(test_timezone(time(NULL), "right/GB"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Australia/West"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Australia/Canberra"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Australia/Broken_Hill"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Australia/Eucla"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Australia/Currie"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Australia/South"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Australia/Lindeman"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Australia/Hobart"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Australia/Perth"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Australia/Yancowinna"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Australia/Victoria"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Australia/Sydney"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Australia/North"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Australia/Adelaide"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Australia/ACT"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Australia/Melbourne"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Australia/Tasmania"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Australia/Darwin"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Australia/Brisbane"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Australia/LHI"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Australia/NSW"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Australia/Queensland"));
      TEST_ASSERT(test_timezone(time(NULL), "right/Australia/Lord_Howe"));
      TEST_ASSERT(test_timezone(time(NULL), "right/NZ"));
      TEST_ASSERT(test_timezone(time(NULL), "Greenwich"));
      TEST_ASSERT(test_timezone(time(NULL), "Eire"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Qatar"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Makassar"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Colombo"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Chungking"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Macau"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Choibalsan"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Rangoon"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Harbin"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Yerevan"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Brunei"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Omsk"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Chongqing"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Tbilisi"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Tehran"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Manila"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Yakutsk"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Qyzylorda"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Dhaka"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Singapore"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Jakarta"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Novosibirsk"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Saigon"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Krasnoyarsk"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Kabul"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Bahrain"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Urumqi"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Thimbu"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Ulan_Bator"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Taipei"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Tashkent"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Dacca"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Aqtau"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Seoul"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Istanbul"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Tel_Aviv"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Almaty"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Phnom_Penh"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Baku"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Anadyr"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Aqtobe"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Kuwait"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Irkutsk"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Ulaanbaatar"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Tokyo"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Gaza"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Riyadh87"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Yekaterinburg"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Riyadh88"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Kolkata"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Nicosia"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Jayapura"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Shanghai"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Pyongyang"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Macao"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Dushanbe"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Kuching"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Vientiane"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Riyadh"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Dili"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Samarkand"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Ashkhabad"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Calcutta"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Hong_Kong"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Sakhalin"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Beirut"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Damascus"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Katmandu"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Jerusalem"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Riyadh89"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Vladivostok"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Kamchatka"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Dubai"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Kashgar"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Ashgabat"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Amman"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Karachi"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Bangkok"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Oral"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Thimphu"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Bishkek"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Baghdad"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Kuala_Lumpur"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Pontianak"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Ho_Chi_Minh"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Ujung_Pandang"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Muscat"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Aden"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Hovd"));
      TEST_ASSERT(test_timezone(time(NULL), "Asia/Magadan"));
      TEST_ASSERT(test_timezone(time(NULL), "EST5EDT"));
      TEST_ASSERT(test_timezone(time(NULL), "PST8PDT"));
      TEST_ASSERT(test_timezone(time(NULL), "Atlantic/Bermuda"));
      TEST_ASSERT(test_timezone(time(NULL), "Atlantic/St_Helena"));
      TEST_ASSERT(test_timezone(time(NULL), "Atlantic/Cape_Verde"));
      TEST_ASSERT(test_timezone(time(NULL), "Atlantic/Stanley"));
      TEST_ASSERT(test_timezone(time(NULL), "Atlantic/Azores"));
      TEST_ASSERT(test_timezone(time(NULL), "Atlantic/Jan_Mayen"));
      TEST_ASSERT(test_timezone(time(NULL), "Atlantic/Reykjavik"));
      TEST_ASSERT(test_timezone(time(NULL), "Atlantic/Madeira"));
      TEST_ASSERT(test_timezone(time(NULL), "Atlantic/Canary"));
      TEST_ASSERT(test_timezone(time(NULL), "Atlantic/Faeroe"));
      TEST_ASSERT(test_timezone(time(NULL), "Atlantic/Faroe"));
      TEST_ASSERT(test_timezone(time(NULL), "Atlantic/South_Georgia"));
      TEST_ASSERT(test_timezone(time(NULL), "Kwajalein"));
      TEST_ASSERT(test_timezone(time(NULL), "UTC"));
      TEST_ASSERT(test_timezone(time(NULL), "GMT-0"));
      TEST_ASSERT(test_timezone(time(NULL), "MST7MDT"));
      TEST_ASSERT(test_timezone(time(NULL), "GB-Eire"));
      TEST_ASSERT(test_timezone(time(NULL), "PRC"));
      TEST_ASSERT(test_timezone(time(NULL), "SystemV/CST6CDT"));
      TEST_ASSERT(test_timezone(time(NULL), "SystemV/CST6"));
      TEST_ASSERT(test_timezone(time(NULL), "SystemV/PST8"));
      TEST_ASSERT(test_timezone(time(NULL), "SystemV/YST9"));
      TEST_ASSERT(test_timezone(time(NULL), "SystemV/EST5"));
      TEST_ASSERT(test_timezone(time(NULL), "SystemV/AST4"));
      TEST_ASSERT(test_timezone(time(NULL), "SystemV/AST4ADT"));
      TEST_ASSERT(test_timezone(time(NULL), "SystemV/YST9YDT"));
      TEST_ASSERT(test_timezone(time(NULL), "SystemV/EST5EDT"));
      TEST_ASSERT(test_timezone(time(NULL), "SystemV/PST8PDT"));
      TEST_ASSERT(test_timezone(time(NULL), "SystemV/MST7MDT"));
      TEST_ASSERT(test_timezone(time(NULL), "SystemV/HST10"));
      TEST_ASSERT(test_timezone(time(NULL), "SystemV/MST7"));
      TEST_ASSERT(test_timezone(time(NULL), "Arctic/Longyearbyen"));
      TEST_ASSERT(test_timezone(time(NULL), "Cuba"));
      TEST_ASSERT(test_timezone(time(NULL), "Israel"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT-3"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT+1"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT-5"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT-13"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT-1"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT-9"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/Zulu"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT+0"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/UCT"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT+12"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT+9"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT-6"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/Universal"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT-2"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/Greenwich"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT+3"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT+8"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/UTC"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT-0"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT-14"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT+10"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT+4"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT+5"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT-12"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT-8"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT+7"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT-11"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT+6"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT0"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT-7"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT+11"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT-4"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT+2"));
      TEST_ASSERT(test_timezone(time(NULL), "Etc/GMT-10"));
      TEST_ASSERT(test_timezone(time(NULL), "HST"));
      TEST_ASSERT(test_timezone(time(NULL), "Iceland"));
      TEST_ASSERT(test_timezone(time(NULL), "Mexico/BajaNorte"));
      TEST_ASSERT(test_timezone(time(NULL), "Mexico/BajaSur"));
      TEST_ASSERT(test_timezone(time(NULL), "Mexico/General"));
      TEST_ASSERT(test_timezone(time(NULL), "Mideast/Riyadh87"));
      TEST_ASSERT(test_timezone(time(NULL), "Mideast/Riyadh88"));
      TEST_ASSERT(test_timezone(time(NULL), "Mideast/Riyadh89"));
      TEST_ASSERT(test_timezone(time(NULL), "Antarctica/Davis"));
      TEST_ASSERT(test_timezone(time(NULL), "Antarctica/DumontDUrville"));
      TEST_ASSERT(test_timezone(time(NULL), "Antarctica/Syowa"));
      TEST_ASSERT(test_timezone(time(NULL), "Antarctica/Palmer"));
      TEST_ASSERT(test_timezone(time(NULL), "Antarctica/Casey"));
      TEST_ASSERT(test_timezone(time(NULL), "Antarctica/Rothera"));
      TEST_ASSERT(test_timezone(time(NULL), "Antarctica/Mawson"));
      TEST_ASSERT(test_timezone(time(NULL), "Antarctica/McMurdo"));
      TEST_ASSERT(test_timezone(time(NULL), "Antarctica/South_Pole"));
      TEST_ASSERT(test_timezone(time(NULL), "Antarctica/Vostok"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Curacao"));
      TEST_ASSERT(test_timezone(time(NULL), "America/St_Lucia"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Managua"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Lima"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Nipigon"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Cayenne"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Detroit"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Kentucky/Louisville"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Kentucky/Monticello"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Belem"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Jujuy"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Godthab"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Guatemala"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Atka"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Montreal"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Resolute"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Thunder_Bay"));
      TEST_ASSERT(test_timezone(time(NULL), "America/North_Dakota/New_Salem"));
      TEST_ASSERT(test_timezone(time(NULL), "America/North_Dakota/Center"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Panama"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Los_Angeles"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Whitehorse"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Santiago"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Iqaluit"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Dawson"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Juneau"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Thule"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Fortaleza"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Montevideo"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Tegucigalpa"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Port-au-Prince"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Guadeloupe"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Coral_Harbour"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Monterrey"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Anguilla"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Antigua"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Campo_Grande"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Buenos_Aires"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Maceio"));
      TEST_ASSERT(test_timezone(time(NULL), "America/New_York"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Puerto_Rico"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Noronha"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Sao_Paulo"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Cancun"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Aruba"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Yellowknife"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Knox_IN"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Halifax"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Grand_Turk"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Vancouver"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Bogota"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Santo_Domingo"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Tortola"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Blanc-Sablon"));
      TEST_ASSERT(test_timezone(time(NULL), "America/St_Thomas"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Scoresbysund"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Jamaica"));
      TEST_ASSERT(test_timezone(time(NULL), "America/El_Salvador"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Montserrat"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Martinique"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Nassau"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Indianapolis"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Pangnirtung"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Port_of_Spain"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Mexico_City"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Denver"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Dominica"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Eirunepe"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Atikokan"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Glace_Bay"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Rainy_River"));
      TEST_ASSERT(test_timezone(time(NULL), "America/St_Barthelemy"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Miquelon"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Indiana/Vevay"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Indiana/Vincennes"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Indiana/Marengo"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Indiana/Petersburg"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Indiana/Tell_City"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Indiana/Knox"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Indiana/Indianapolis"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Indiana/Winamac"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Menominee"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Porto_Velho"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Phoenix"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Argentina/San_Juan"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Argentina/Jujuy"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Argentina/Ushuaia"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Argentina/Buenos_Aires"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Argentina/La_Rioja"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Argentina/ComodRivadavia"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Argentina/Tucuman"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Argentina/San_Luis"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Argentina/Rio_Gallegos"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Argentina/Mendoza"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Argentina/Cordoba"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Argentina/Catamarca"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Dawson_Creek"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Merida"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Moncton"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Goose_Bay"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Grenada"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Barbados"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Tijuana"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Regina"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Ensenada"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Louisville"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Edmonton"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Bahia"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Nome"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Guayaquil"));
      TEST_ASSERT(test_timezone(time(NULL), "America/La_Paz"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Costa_Rica"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Mazatlan"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Havana"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Marigot"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Mendoza"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Virgin"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Manaus"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Rosario"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Boa_Vista"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Winnipeg"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Paramaribo"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Danmarkshavn"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Caracas"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Swift_Current"));
      TEST_ASSERT(test_timezone(time(NULL), "America/St_Johns"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Araguaina"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Adak"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Cordoba"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Fort_Wayne"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Catamarca"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Recife"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Toronto"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Anchorage"));
      TEST_ASSERT(test_timezone(time(NULL), "America/St_Vincent"));
      TEST_ASSERT(test_timezone(time(NULL), "America/St_Kitts"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Chihuahua"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Cayman"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Belize"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Cambridge_Bay"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Cuiaba"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Chicago"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Guyana"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Inuvik"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Asuncion"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Porto_Acre"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Hermosillo"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Shiprock"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Rio_Branco"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Yakutat"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Rankin_Inlet"));
      TEST_ASSERT(test_timezone(time(NULL), "America/Boise"));
      TEST_ASSERT(test_timezone(time(NULL), "Brazil/West"));
      TEST_ASSERT(test_timezone(time(NULL), "Brazil/Acre"));
      TEST_ASSERT(test_timezone(time(NULL), "Brazil/East"));
      TEST_ASSERT(test_timezone(time(NULL), "Brazil/DeNoronha"));
      TEST_ASSERT(test_timezone(time(NULL), "GMT0"));
      TEST_ASSERT(test_timezone(time(NULL), "Libya"));
      TEST_ASSERT(test_timezone(time(NULL), "W-SU"));
      TEST_ASSERT(test_timezone(time(NULL), "NZ-CHAT"));
      TEST_ASSERT(test_timezone(time(NULL), "Factory"));
      TEST_ASSERT(test_timezone(time(NULL), "GB"));
      TEST_ASSERT(test_timezone(time(NULL), "Australia/West"));
      TEST_ASSERT(test_timezone(time(NULL), "Australia/Canberra"));
      TEST_ASSERT(test_timezone(time(NULL), "Australia/Broken_Hill"));
      TEST_ASSERT(test_timezone(time(NULL), "Australia/Eucla"));
      TEST_ASSERT(test_timezone(time(NULL), "Australia/Currie"));
      TEST_ASSERT(test_timezone(time(NULL), "Australia/South"));
      TEST_ASSERT(test_timezone(time(NULL), "Australia/Lindeman"));
      TEST_ASSERT(test_timezone(time(NULL), "Australia/Hobart"));
      TEST_ASSERT(test_timezone(time(NULL), "Australia/Perth"));
      TEST_ASSERT(test_timezone(time(NULL), "Australia/Yancowinna"));
      TEST_ASSERT(test_timezone(time(NULL), "Australia/Victoria"));
      TEST_ASSERT(test_timezone(time(NULL), "Australia/Sydney"));
      TEST_ASSERT(test_timezone(time(NULL), "Australia/North"));
      TEST_ASSERT(test_timezone(time(NULL), "Australia/Adelaide"));
      TEST_ASSERT(test_timezone(time(NULL), "Australia/ACT"));
      TEST_ASSERT(test_timezone(time(NULL), "Australia/Melbourne"));
      TEST_ASSERT(test_timezone(time(NULL), "Australia/Tasmania"));
      TEST_ASSERT(test_timezone(time(NULL), "Australia/Darwin"));
      TEST_ASSERT(test_timezone(time(NULL), "Australia/Brisbane"));
      TEST_ASSERT(test_timezone(time(NULL), "Australia/LHI"));
      TEST_ASSERT(test_timezone(time(NULL), "Australia/NSW"));
      TEST_ASSERT(test_timezone(time(NULL), "Australia/Queensland"));
      TEST_ASSERT(test_timezone(time(NULL), "Australia/Lord_Howe"));
      TEST_ASSERT(test_timezone(time(NULL), "NZ"));
    }
#endif
  app_shutdown();
  return 0;
}
