package apache;
#
#  Package:      apache.pm
#
#  Copyright:     COPYRIGHT Compaq Computer Corporation 2001.  All rights reserved.
#
#  Description:  The OS specific routines for changing the system.
#
#    These routines handle APACHE specific changes including:
#
#     - Apache configuration
#     - Virtual site creation/editing
#     - Stop/Start of Apache daemon
#     - etc.
#
#  History:
#
use strict;

use CGI;
use XML::DOM;
use XML::QL;
use Quota;
use Time::localtime;

use custom::groups;
use custom::users;
use custom::sendmail;

use config;
use general;
use datastore;
use message;
use error;


# A sub-routine that deletes the virtual site from the system
sub delVirtualSite {

    my %system = config::system();

    my $sitename = shift || return(0);

    my $groupID = "";
    my $siteadmin = "";
    
    # Remove the site from the XML
    # Delete it from the XML
    # Parse the virtual sites file
    my $parser = new XML::DOM::Parser;
    my $doc = $parser->parsefile("$system{'datastore'}/virtualsites.xml");

    my $virtualSites = eval { $doc->getElementsByTagName("VirtualHost"); } || "";
    my $numOfVirtualSites = eval { $virtualSites->getLength(); } || "0";
    
    # Find the correct virtual site
    for (my $i=0; $i<$numOfVirtualSites; $i++) {
	
        my $thisServerName = $virtualSites->item($i)->getElementsByTagName("ServerName")->item(0)->getFirstChild->getData();
	
	# If we found the right site, delete it
	if ($sitename eq $thisServerName) {

	    # Now remove the virtual site from the datastore
	    $groupID = eval { $virtualSites->item($i)->getElementsByTagName("Group")->item(0)->getFirstChild->getData(); } || "";
	    $siteadmin = eval { $virtualSites->item($i)->getElementsByTagName("ServerAdmin")->item(0)->getFirstChild->getData(); } || "";
            $doc->getFirstChild->removeChild($virtualSites->item($i));
            last;
	}
    }

    general::debug( "About to delete virtual site group virtualsite_$groupID");

    my %userInfo = datastore::userInfoByID($siteadmin);
    users::delUserFromOSGroup($userInfo{'loginname'}, "site_admins");

    # Parse the virtual sites file
    my $parser2 = new XML::DOM::Parser;
    my $grpdoc = $parser2->parsefile("$system{'datastore'}/groups.xml");

    # Remove symbolic links
    my $command = "/bin/cat /etc/group | /bin/grep ^virtualsite_$groupID:";
    my $result = `$command`;
    $result =~ s/\n//g;

    my @pieces = split(":", $result);
    my @members = split(",", $pieces[3]);
    
    foreach my $member (@members) {
	$command = "/bin/rm /home/$member/$sitename";
	$result = `$command`;

	general::debug( "Command = $command\nResult = $result");
    }

    # Remove sym link for primary administrator
    my $command = "/bin/rm /home/$userInfo{'loginname'}/$sitename";
    my $result = `$command`;

  datastore::delGroup($grpdoc, "virtualsite_$groupID");

  datastore::XMLtoFile($grpdoc, "$system{'datastore'}/groups.xml");

  datastore::XMLtoFile($doc, "$system{'datastore'}/virtualsites.xml");

    # Remove the virtual site group
    $command = "/usr/sbin/groupdel virtualsite_$groupID";
    $result = `$command`;

    $command = "/usr/sbin/grpconv";
    $result = `$command`;

    # Remove the directory
    # This is extreme prejudice...need to add options for backing up
    $command = "/bin/rm -rf $system{'vhostroot'}/$sitename";
    $result = `$command`;

    $command = "/bin/rm /etc/logrotate.d/$sitename";
    $result = `$command`;

    $command = "/bin/rm /usr/local/frontpage/$sitename:80.cnf";
    $result = `$command`;

    $command = "/bin/rm $system{'root'}/system/reports/webalizer_$sitename.conf";
    $result = `$command`;

    # Remove the virtual site from the vhost.conf file
    if (open(VHOSTFILE, "<$system{'vhostsfile'}")) {

	my @lines = <VHOSTFILE>;

	close(VHOSTFILE);

	my $inVirtualHost = 0;
	my @newlines = ();

	# Strip out this virtual site
	foreach my $line (@lines) {
	    
	    if ($line =~ m/ServerName $sitename/) {
		pop(@newlines);
		$inVirtualHost = 1;
	    }
	    elsif ($line =~ m/<\/VirtualHost>/ && $inVirtualHost == 1) {
		$inVirtualHost = 0;
		next;
	    }
	    elsif ($line =~ m/<VirtualHost/ && $inVirtualHost == 1) {
		push(@newlines,$line);
		$inVirtualHost = 0;
	    }
	    elsif ($line =~ m/ServerName/ && $inVirtualHost == 1) {
		push(@newlines,$line);
		$inVirtualHost = 0;
	    }
	    elsif ($inVirtualHost == 1) {
		next;
	    }
	    else {
		push(@newlines, $line);
	    }
	    
	}

	if (open(VHOSTFILE, ">$system{'vhostsfile'}")) {

	    foreach my $line (@newlines) {
		print VHOSTFILE $line;
	    }

	    close(VHOSTFILE);

	}

    }

    # Update sendmail generics domain
    sendmail::updateSendmailGenericsDomain();

    # Update the e-mail aliases
    sendmail::updateSendmail();

    # Restart Apache
    restartApache();

}


# A sub-routine that updates the IP address in vhosts.conf
sub updateVirtualSiteIPs {

    my $oldIP = shift || return;
    my $newIP = shift || return;

    my %system = config::system;

    general::debug( "Updating virtual site IPs - Old IP = $oldIP");
    
    # First grab all IP addresses

    my $parser = new XML::DOM::Parser;
    my $doc = $parser->parsefile("$system{'datastore'}/network.xml");

    my $interfaces = $doc->getElementsByTagName("Interfaces")->item(0)->getElementsByTagName("Interface",0);
    my $numOfInterfaces = $interfaces->getLength;

    my $array = "";

    for (my $j=0; $j<$numOfInterfaces;$j++) {

	my $interfaceNode = $interfaces->item($j);
	my $ip1 = eval {$interfaceNode->getElementsByTagName("IP")->item(0)->getFirstChild->getData();} || "";

	$array .= "$ip1<TD>";
    }
    
    # Split out the values
    my @ips = split(/<TD>/,$array);
    my $iplist = join(' ', @ips);

    # If the vhosts file does not exists, populate it with the root web
    if  ((-s $system{'vhostsfile'}) <= 0) {

	`/bin/rm /usr/local/frontpage/we80.cnf`;

	open(VHOSTFILE, ">$system{'vhostsfile'}");

	print VHOSTFILE "<VirtualHost $iplist>
   DocumentRoot \"/var/www/html\"\n";

	my $fpextensions = `/bin/cat $system{'datastore'}/apache.xml | /bin/grep \\\<FrontPageExtensions\\\>Yes`;

	# If front page is on, add it to the root web
	if ($fpextensions ne "") {
	    
	    print VHOSTFILE "
  <Directory /var/www/html>
    AllowOverride All
  </Directory>

  <Location \"/_vti_pvt\">
    Order deny,allow
    Deny from all
  </Location>
  <Location \"/_vti_cnf\">
    Order deny,allow
    Deny from all
  </Location>

<IfDefine HAVE_FRONTPAGE>
  ScriptAlias /_vti_bin/_vti_adm/ /var/www/html/_vti_b\in/_vti_adm/
  ScriptAlias /_vti_bin/_vti_aut/ /var/www/html/_vti_b\in/_vti_aut/
  ScriptAlias /_vti_bin/ /var/www/html/_vti_bin/
</IfDefine>

";

	}

	print VHOSTFILE "</VirtualHost>\n";

	foreach my $ip (@ips) {
	    print VHOSTFILE "NameVirtualHost $ip\n";
	}

	close(VHOSTFILE);

    }

    # Read in the current vhosts.conf file
    if (open(VHOSTFILE, "<$system{'vhostsfile'}")) {
	
	my @lines = <VHOSTFILE>;
	close(VHOSTFILE);
	
	# Overwrite the current virtual sites file
	if (open(VHOSTFILE, ">$system{'vhostsfile'}")) {
	    
	    my $printedNameVirtualHost = 0;

	    foreach my $line (@lines) {

		if ($line =~ m/^NameVirtualHost/ && $printedNameVirtualHost == 0) {
		    $printedNameVirtualHost = 1;
		    foreach my $ip (@ips) {
			print VHOSTFILE "NameVirtualHost $ip\n";
		    }
		}
		elsif ($line =~ m/^NameVirtualHost/) {
		    next;
		}
		# Replace virtual host directive with new IP address
		elsif ($line =~ m/^<VirtualHost/) {
		  general::debug("OldIP = '$oldIP' IP = '$newIP'");
		    if ($oldIP ne "") {
			$line =~ s/$oldIP/$newIP/;
		    }
		    print VHOSTFILE $line;
		}
		else {
		    print VHOSTFILE $line;
		}
	    }
	}
    }
	
    return(1);

}

# A sub-routine that updates the virtual host entries
sub updateVirtualSite {

    my $sitename = shift || "";
    my $oldsiteadmin = shift || "";

#    general::debug( "    RECEIVED OLD SITE ADMIN OF $oldsiteadmin");

    my %system = config::system;

    # First grab all IP addresses

    my $netparser = new XML::DOM::Parser;
    my $netdoc = $netparser->parsefile("$system{'datastore'}/network.xml");

    my $interfaces = $netdoc->getElementsByTagName("Interfaces")->item(0)->getElementsByTagName("Interface",0);
    my $numOfInterfaces = $interfaces->getLength;

    my $array = "";

    for (my $j=0; $j<$numOfInterfaces;$j++) {

	my $interfaceNode = $interfaces->item($j);
	my $ip1 = eval {$interfaceNode->getElementsByTagName("IP")->item(0)->getFirstChild->getData();} || "";

	$array .= "$ip1<TD>";
    }
    
    # Split out the values
    my @ips = split(/<TD>/,$array);
    my $iplist = join(' ', @ips);
    general::debug( "Updating virtual host file");

    my $newVhostFile = "";

    # Parse the virtual sites
    my $parser1 = new XML::DOM::Parser;
    my $doc1 = $parser1->parsefile("$system{'datastore'}/virtualsites.xml");
    
    # Search for this virtual site
    my $foundNode = "";
    for my $node ($doc1->getElementsByTagName("VirtualHost")) {

	my $thisSiteName = eval { $node->getElementsByTagName("ServerName")->item(0)->getFirstChild->getData(); } || "";

	if ($thisSiteName eq $sitename) {
	    $foundNode = $node;
	    last;
	}
    }

    if ($foundNode eq "") {
	general::debug( "Attempted virtual site update on non-existant site!");
	return(1);
    }
    else {


	my $docroot = $foundNode->getElementsByTagName("DocumentRoot")->item(0)->getFirstChild->getData(); 
	my $directory = $docroot;
	$directory =~ s/\/html//;

	my $siteadmin = $foundNode->getElementsByTagName("ServerAdmin")->item(0)->getFirstChild->getData();

	my %adminInfo = datastore::userInfoByID($siteadmin);
	my %oldAdminInfo = datastore::userInfoByID($oldsiteadmin);

	general::debug( "  OLDSITE ADMIN LOGINNAME = $oldAdminInfo{'loginname'}");

	# Update the site_admins group here

#	general::debug( "   **** Updating site admins: deleting $oldAdminInfo{'loginname'} and adding $adminInfo{'loginname'}");

        users::delUserFromOSGroup($oldAdminInfo{'loginname'}, "site_admins");
	users::addUserToOSGroup($adminInfo{'loginname'}, "site_admins");

	my $frontpage = eval { $foundNode->getElementsByTagName("AllowFrontPage")->item(0)->getFirstChild->getData(); } || "";	
	my $group = eval { $foundNode->getElementsByTagName("Group")->item(0)->getFirstChild->getData(); } || "";
	my $status = eval { $foundNode->getElementsByTagName("Status")->item(0)->getFirstChild->getData(); } || "";
	my $cgi = eval { $foundNode->getElementsByTagName("AllowCGI")->item(0)->getFirstChild->getData(); } || "";
	my $frontpage = eval { $foundNode->getElementsByTagName("AllowFrontPage")->item(0)->getFirstChild->getData(); } || "";
	my $php = eval { $foundNode->getElementsByTagName("AllowPHP")->item(0)->getFirstChild->getData(); } || "";
	my $modperl = eval { $foundNode->getElementsByTagName("AllowModPerl")->item(0)->getFirstChild->getData(); } || "";
	my $bandwidth = eval { $foundNode->getElementsByTagName("QOS")->item(0)->getElementsByTagName("Bandwidth")->item(0)->getFirstChild->getData(); } || "";
	my $limit_cpu = eval { $foundNode->getElementsByTagName("Limits")->item(0)->getElementsByTagName("CPU")->item(0)->getFirstChild->getData(); } || "";
	my $limit_memory = eval { $foundNode->getElementsByTagName("Limits")->item(0)->getElementsByTagName("Memory")->item(0)->getFirstChild->getData(); } || "";
	my $limit_procs = eval { $foundNode->getElementsByTagName("Limits")->item(0)->getElementsByTagName("Processes")->item(0)->getFirstChild->getData(); } || "";

	# Make sure the site administrator is a part of the virtual site group
	users::addUserToOSGroup($adminInfo{'loginname'}, "virtualsite_$group");

	my $ip = eval { $foundNode->getAttribute("ip"); } || "";
	if ($ip eq "" || $ip eq "default") { $ip = "$iplist"; }

	# Remove the virtual site from the vhost.conf file
	if (open(VHOSTFILE, "<$system{'vhostsfile'}")) {
	    
	    my @lines = <VHOSTFILE>;
	    
	    close(VHOSTFILE);
	    
	    my $inVirtualHost = 0;
	    my @newlines = ();
	    
	    # Strip out this virtual site
	    foreach my $line (@lines) {
		
		if ($line =~ m/ServerName $sitename/) {
		    pop(@newlines);
		    $inVirtualHost = 1;
		}
		elsif ($line =~ m/<\/VirtualHost>/ && $inVirtualHost == 1) {
		    $inVirtualHost = 0;
		    next;
		}
		elsif ($line =~ m/<VirtualHost/ && $inVirtualHost == 1) {
		    push(@newlines,$line);
		    $inVirtualHost = 0;
		}
		elsif ($line =~ m/ServerName/ && $inVirtualHost == 1) {
		    push(@newlines,$line);
		    $inVirtualHost = 0;
		}
		elsif ($inVirtualHost == 1) {
		    next;
		}
		else {
		    push(@newlines, $line);
		}
	    
	    }
	    
	    if (open(VHOSTFILE, ">$system{'vhostsfile'}")) {
		
		foreach my $line (@newlines) {
		    print VHOSTFILE $line;
		}

		# Add the new one if active
		if ($status eq "Active") {
		    
		    print VHOSTFILE "
<VirtualHost $ip>
  ServerName $sitename
  CustomLog $directory/logs/access_log common
  LogLevel warn
  ErrorLog $directory/logs/error_log
  DocumentRoot $docroot

  <Directory $docroot>
    AllowOverride All
  </Directory>
";

		    # Setup basic cgi
		    if ($cgi eq "Yes") {

			print VHOSTFILE "
  ScriptAlias /cgi-bin/ \"$directory/cgi-bin/\"

  <Directory \"$directory/cgi-bin\">
    AllowOverride None
    Options ExecCGI
    Order allow,deny
    Allow from all
  </Directory>

";
		    }

		    # Deny access to php served pages
		    if ($php ne "Yes") {
			print VHOSTFILE "<Files ~ \"\\.*(php3|php4|php|phtml)\">
   Order allow,deny
   Deny from all
</Files>

";
		    }

		    if ($modperl eq "Yes") {

			# Setup mod_perl
			print VHOSTFILE "

<IfModule mod_perl.c>
  ScriptAlias /mod_perl/ \"$directory/mod_perl/\"

  <Location /mod_perl>
    SetHandler perl-script
    PerlHandler Apache::Registry
    Options +ExecCGI
  </Location>
</IfModule>
";
		    }

		    # Setup mod_throttle
		    # if ($bandwidth ne "" && $bandwidth ne "0") {
		    #print VHOSTFILE "
#  ThrottlePolicy Volume $bandwidth 1s\n";
#		    }

		    # Setup DoS Limits
		    if ($limit_cpu ne "") {
			print VHOSTFILE "
  RLimitCPU     $limit_cpu";
		    }
		    if ($limit_procs ne "") {
			print VHOSTFILE "
  RLimitNPROC     $limit_procs";
		    }
		    if ($limit_memory ne "") {
			print VHOSTFILE "
  RLimitMEM     $limit_memory";
		    }
		    
		    # Setup Front Page Extensions
		    print VHOSTFILE "
  <Location \"/_vti_pvt\">
    Order deny,allow
    Deny from all
  </Location>
  <Location \"/_vti_cnf\">
    Order deny,allow
    Deny from all
  </Location>
";

		    if ($frontpage eq "Yes") {

			print VHOSTFILE "

<IfDefine HAVE_FRONTPAGE>
  ScriptAlias /_vti_bin/_vti_adm/ /home/VirtualSites/$sitename/html/_vti_b\in/_vti_adm/
  ScriptAlias /_vti_bin/_vti_aut/ /home/VirtualSites/$sitename/html/_vti_b\in/_vti_aut/
  ScriptAlias /_vti_bin/ /home/VirtualSites/$sitename/html/_vti_bin/
</IfDefine>
";

		    }
		    else {
			print VHOSTFILE "
  <Location \"/_vti_bin\">
    Order deny,allow
    Deny from all
  </Location>
";
		    }

		    print VHOSTFILE "
</VirtualHost>\n";

		}

	    }

	    close(VHOSTFILE);
	    
	}

	# Make sure the site administrator is a part of the virtual site group
        users::addUserToOSGroup($siteadmin, "virtualsite_$group");

	# Reset permissions
	my $command = "/bin/chown -R apache:$group $directory";
	my $result = `$command`;
	$command = "/bin/chmod g+ws $directory";
	$result = `$command`;
	$command = "/bin/chown -R $siteadmin:$group $directory/html";
	$result = `$command`;
	$command = "/bin/chmod g+ws $directory/html";
	$result = `$command`;
	$command = "/bin/chown -R $siteadmin:$group $directory/logs";
	$result = `$command`;	
	$command = "/bin/chmod o-rwx $directory/logs";
	$result = `$command`;
	$command = "/bin/chown -R $siteadmin:$group $directory/cgi-bin";
	$result = `$command`;
	$command = "/bin/chmod g+ws $directory/cgi-bin";
	$result = `$command`;
	$command = "/bin/chown -R $siteadmin:$group $directory/mod_perl";
	$result = `$command`;	
	$command = "/bin/chmod g+ws $directory/mod_perl";
	$result = `$command`;

	# Fix up front page permissions
	$command = "$system{'root'}/system/bin/chmod ug+ws $docroot/_vti_bin/shtml.exe";
	$result = `$command`;
	$command = "$system{'root'}/system/bin/chmod ug+ws $docroot/_vti_bin/fpcount.exe";
	$result = `$command`;
	$command = "$system{'root'}/system/bin/chmod ug+ws $docroot/_vti_bin/_vti_adm/admin.exe";
	$result = `$command`;
	$command = "$system{'root'}/system/bin/chmod ug+ws $docroot/_vti_bin/_vti_aut/author.exe";
	$result = `$command`;

	# Update Front Page password file
	$result = `/bin/rm $docroot/_vti_pvt/service.pwd`;
	$result = `/bin/rm $docroot/_vti_pvt/service.grp`;

        my %userInfo = datastore::userInfoByID($siteadmin);

        # Update the password files
	# Rebuild the service.grp file
        if (open (FPFILE, ">$docroot/_vti_pvt/service.grp")) {
            print FPFILE "# -FrontPage-\n";
            print FPFILE "administrators: $userInfo{'loginname'}\n";

            print FPFILE "authors: ";

            close(FPFILE);
        }

	# Rebuild the service.pwd file
        if (open (FPFILE, ">$docroot/_vti_pvt/service.pwd")) {
            print FPFILE "# -FrontPage-\n";
	    close(FPFILE);
	}

	updateHTPasswd ($userInfo{'loginname'}, "$docroot/_vti_pvt/service.pwd");

	# Fix-up permissions
	$command = "/bin/chown -R $siteadmin:$group $directory/html/_vti_pvt/service.pwd";
	$result = `$command`;	
	$command = "/bin/chown -R $siteadmin:$group $directory/html/_vti_pvt/service.grp";
	$result = `$command`;

	# Add symbolic links
	my $members = `/bin/cat /etc/group | /bin/grep ^virtualsite_$group`;
	my @pieces = split(":", $members);
	my @members = split(",", $pieces[3]);

	foreach my $member (@members) {

	    $member =~ s/\n//g;

	    $result = `/bin/ln -s $directory /home/$member/$sitename`;
	    $result = `/bin/chown $member:$member /home/$member/$sitename`;

	}

    }

    general::debug( "Returning from updateVirtualSite");

    return(1);

}

# A sub-routine that applies a profile to the apache configuration
sub applyApacheProfile {

    # Step 1:  Grab profile id and all applied items
    my $profileid = shift || return;
    my $itemsOn = join(' ', @_);
    $itemsOn = " $itemsOn ";

    my %system = config::system();

    # Step 1a:  Update the general.xml to store selected profile
    my $genparser = new XML::DOM::Parser;
    my $gendoc = $genparser->parsefile("$system{'datastore'}/general.xml");

  datastore::textNodeUpdate($gendoc->getElementsByTagName("SystemProfile")->item(0), $profileid);

    $gendoc->printToFile("$system{'datastore'}/general.xml");

    $gendoc->dispose();

    # Step 2:  Read profiles information and get the detail for this one
    my $parser = new XML::DOM::Parser;
    my $doc = $parser->parsefile("$system{'datastore'}/profiles.xml");

    my $foundNode = "";
    foreach my $node ($doc->getElementsByTagName("Profile")) {

	my $thisid = $node->getAttribute("id");

	if ($thisid eq $profileid) {
	    $foundNode = $node;
	    last;
	}
    }

    if ($foundNode eq "") {
	$doc->dispose();
	return(0);
    }

    # Step 3:  Update apache.xml
    my $items = $foundNode->getElementsByTagName("Item");
    my $numOfItems = $items->getLength();

    my $apacheDaemon = "Standard";

    general::debug( "Number of Items = $numOfItems");

    for (my $k=0; $k<$numOfItems; $k++) {

	my $node = $items->item($k);
	my $itemid = eval { $node->getAttribute("id"); } || "";

#	general::debug( "Item ID = $itemid");

	if ($itemid <= 0) {
	    
	    $apacheDaemon = eval { $node->getElementsByTagName("Apache")->item(0)->getFirstChild->getData(); } || "Standard";
	    general::debug( "Skipping sub zero item detail");
	    next;
	}

	# If this item is on, then make the change
	if ($itemsOn =~ m/ $itemid /) {

	    foreach my $fileNode ($node->getElementsByTagName("File")) {

		my $name = $fileNode->getAttribute("name");

#		general::debug( "FILE = $name");

		my $parser2 = new XML::DOM::Parser;
		my $doc2 = $parser->parsefile("$system{'datastore'}/$name");

		# Step 3a:  For each node in the detail, update the actual file
		# WARNING:  This is not generally applicable as a DOM tree merge.  It works
		# on the assumption that tags are strictly unique within the document.
		# Don't we wish everything was made like rubbermaid?  :-)
		foreach my $change ($fileNode->getChildNodes()) {

		    my @children = $change->getChildNodes();
		    my $numOfChildren = scalar(@children);

		    for (my $i=0; $i<$numOfChildren; $i++) {
			
			my $changeTag = eval { $children[$i]->getNodeName(); } || "";
#			general::debug( "$i $itemid in file $name Found change node $changeTag");

			if ($changeTag ne "#text#") {
			    my $changeValue = eval { $children[$i]->getFirstChild->getData(); } || "";
			    general::debug( "  - Change value = $changeValue");
			  datastore::textNodeUpdate($doc2->getElementsByTagName("$changeTag")->item(0), $changeValue);
			}
		    }
		}

		$doc2->printToFile("$system{'datastore'}/$name");

		$doc2->dispose();

	    }
	}

    }

    # Step 4:  Move httpd daemon (if necessary), DSO's, and config files
    stopApache();

    # Give stop apache some time to work
    sleep(3);

    # Give the web server time to die
    my $httpdresult = `/bin/ps auwx | /bin/grep /usr/sbin/httpd | /usr/bin/wc -l`;
    $httpdresult =~ s/\W//g;
    $httpdresult = scalar($httpdresult);

    general::debug( "HTTPDRESULTS = $httpdresult");

    my $count = 0;

    while ($httpdresult > 3 && $count < 6) {
	++$count;
	sleep(1);
	$httpdresult = `/bin/ps auwx | /bin/grep /usr/sbin/httpd | /usr/bin/wc -l`;
	$httpdresult =~ s/\W//g;
	$httpdresult = scalar($httpdresult);
	general::debug( "Sleeping 1 second - $httpdresult");
    }

    general::debug( "Daemon = $apacheDaemon");
    if ($apacheDaemon eq "Standard") {
	general::debug( `/bin/cp /usr/compaq/bin/apache/apachestd/httpd /usr/sbin/httpd`);
    }
    elsif ($apacheDaemon eq "Optimized") {
	general::debug( `/bin/cp /usr/compaq/bin/apache/apacheopt/httpd /usr/sbin/httpd`);
    }

    # Step 5:  Cycle Apache
    updateApache();

}

# A sub-routine that updates the httpd.conf file based on the XML configuration
sub updateApache {

    my $oldServerAdmin = shift || "";

    my %system = config::system;

    # Release 16 - This is a patch to correct fphttpd.conf
    if ( -e "$system{'root'}/system/fphttpd.conf" ) {
	updateVirtualSitesFPConf();
	`/bin/rm $system{'root'}/system/fphttpd.conf`;
    }



    my $configFile = $system{'httpd.conf'};

    # Parse the apache file
    my $parser = new XML::DOM::Parser;
    my $apachedoc = $parser->parsefile("$system{'datastore'}/apache.xml");

    # O.K. Step #1 - Read existing httpd.conf
    open(FILE, "<$configFile");
    my @lines = <FILE>;
    my $numOfLines = scalar(@lines);
    close(FILE);

    # Step #2 - General, Global Configuration Information
    for (my $i=0; $i<$numOfLines; $i++) {

	# Skip things that break this logic
	if ($lines[$i] =~ m/mod_foo.so/) {
	    next;
	}

	# Load/Un-load modules
	if ($lines[$i] =~ m/^LoadModule/) {

  	    # Extract module name using split
  	    my @mods = split(" ", $lines[$i]);
           
  	    # Check if it exists in the XML
	    my $checkXML = datastore::ApacheCheckModule($apachedoc,$mods[1]);

  	    # If it does exist in XML, then leave the line alone
  	    # If it does not exist in XML, then comment it out
  	    if ($checkXML != 1) {
 		$lines[$i] =~ s/^\s*LoadModule/# LoadModule/;
  	    }

	    next;

  	}
  	elsif ($lines[$i] =~ m/^\#\s*LoadModule/) {
	    
  	    # Extract module name using split
  	    my @mods = split(" ", $lines[$i]);
	    
  	    # Check if it exists in the XML
  	    my $checkXML = datastore::ApacheCheckModule($apachedoc,$mods[2]);
	    
  	    # If it does exist in XML, then uncomment line
  	    # If it does not exist in XML, leave it alone
	    
  	    if ($checkXML == 1) {
  		$lines[$i] =~ s/^#\s*LoadModule/LoadModule/;
  	    }

	    next;

  	}
	# Check for corresponding add module directive
	elsif ($lines[$i] =~ m/^AddModule/) {

  	    # Extract module name using split
  	    my @mods = split(" ", $lines[$i]);
           
  	    # Check if it exists in the XML
	    my $checkXML = datastore::ApacheCheckModule($apachedoc,$mods[1]);
           
  	    # If it does exist in XML, then leave the line alone
  	    # If it does not exist in XML, then comment it out
  	    if ($checkXML != 1) {
 		$lines[$i] =~ s/^\s*AddModule/# AddModule/;
  	    }

	    next;

  	}
  	elsif ($lines[$i] =~ m/^\#\s*AddModule/) {
	    
  	    # Extract module name using split
  	    my @mods = split(" ", $lines[$i]);
	    
  	    # Check if it exists in the XML
  	    my $checkXML = datastore::ApacheCheckModule($apachedoc,$mods[2]);
	    
  	    # If it does exist in XML, then uncomment line
  	    # If it does not exist in XML, leave it alone
	    
  	    if ($checkXML == 1) {
  		$lines[$i] =~ s/^#\s*AddModule/AddModule/;
  	    }

	    next;

  	}
	# Admin E-Mail
	if ($lines[$i] =~ m/^\s*#ServerAdmin/ || $lines[$i] =~ m/^\s*ServerAdmin/) {

	    my $adminemail = eval { $apachedoc->getElementsByTagName("ServerAdminEmail")->item(0)->getFirstChild->getData(); } || "";
	    $adminemail =~ s/ //g;

	    if ($adminemail ne "") {
		$lines[$i] = "ServerAdmin $adminemail\n";
	    }
	    else {
		$lines[$i] = "#ServerAdmin\n";
	    }
	}
	
	# Server Name
	elsif ($lines[$i] =~ m/^\s*#ServerName/ || $lines[$i] =~ m/^\s*ServerName/) {

	    my $servname = eval { $apachedoc->getElementsByTagName("ServerName")->item(0)->getFirstChild->getData(); } || "";
		
	    if ($servname ne "") {
		$lines[$i] = "ServerName $servname\n";
	    }
	    else {
		$lines[$i] = "#ServerName localhost\n";
	    }

	}
#  	# LockFile
#  	elsif ($lines[$i] =~ m/^\s*LockFile/) {
#  	    my $lockfile = eval { $apachedoc->getElementsByTagName("LockFile")->item(0)->getFirstChild->getData(); } || "";
#  	    if ($lockfile ne "") {
#  		$lines[$i] = "LockFile $lockfile\n";
#  	    }
#  	}
	
#  	# PidFile
#  	elsif ($lines[$i] =~ m/^\s*PidFile/) {
#  	    my $pidfile = eval { $apachedoc->getElementsByTagName("PidFile")->item(0)->getFirstChild->getData(); } || "";
#  	    if ($pidfile ne "") {
#  		$lines[$i] = "PidFile $pidfile\n";
#  	    }
#  	}

#  	# ScoreBoardFile
#  	elsif ($lines[$i] =~ m/^\s*ScoreBoardFile/) {
#  	    my $scoreboardfile = eval { $apachedoc->getElementsByTagName("ScoreBoardFile")->item(0)->getFirstChild->getData(); } || "";
#  	    $lines[$i] = "ScoreBoardFile $scoreboardfile\n";
#  	}
	
	# DocumentRoot
	elsif ($lines[$i] =~ m/^\s*DocumentRoot/) {
	    my $apachedocroot = eval { $apachedoc->getElementsByTagName("DocumentRoot")->item(0)->getFirstChild->getData(); } || "";
	    $lines[$i] = "DocumentRoot $apachedocroot\n";
	}

	# Minimum # of Spare Servers
	elsif ($lines[$i] =~ m/^\s*MinSpareServers/) {
	    my $minserv = eval { $apachedoc->getElementsByTagName("MinSpareServers")->item(0)->getFirstChild->getData(); } || "";
	    if ($minserv ne "") {
		$lines[$i] = "MinSpareServers $minserv\n";
	    }
	    else {
		$lines[$i] = "MinSpareServers 5\n";
	    }
	}
	# Maximum # of Spare Servers
	elsif ($lines[$i] =~ m/^\s*MaxSpareServers/) {
	    my $maxserv = eval { $apachedoc->getElementsByTagName("MaxSpareServers")->item(0)->getFirstChild->getData(); } || "";
	    if ($maxserv ne "") {
		$lines[$i] = "MaxSpareServers $maxserv\n";
	    }
	    else {
		$lines[$i] = "MaxSpareServers 20\n";
	    }
	}

	# Number of Start Servers
	elsif ($lines[$i] =~ m/^\s*StartServers/) {
	    my $strtserv = eval { $apachedoc->getElementsByTagName("StartServers")->item(0)->getFirstChild->getData(); } || "";
	    if ($strtserv ne "") {
		$lines[$i] = "StartServers $strtserv\n";
	    }
	    else {
		$lines[$i] = "StartServers 8\n";
	    }
	}

	# Maximum # of Clients
	elsif ($lines[$i] =~ m/^\s*MaxClients/) {
	    my $maxclients = eval { $apachedoc->getElementsByTagName("MaxClients")->item(0)->getFirstChild->getData(); } || "";
	    if ($maxclients ne "") {
		$lines[$i] = "MaxClients $maxclients\n";
	    }
	    else {
		$lines[$i] = "MaxClients 150\n";
	    }
	}

	# Maximum # of Requests per Child
	elsif ($lines[$i] =~ m/^\s*MaxRequestsPerChild/) {
	    my $maxrequests = eval { $apachedoc->getElementsByTagName("MaxRequestsPerChild")->item(0)->getFirstChild->getData(); } || "";
	    if ($maxrequests ne "") {
		$lines[$i] = "MaxRequestsPerChild $maxrequests\n";
	    }
	    else {
		$lines[$i] = "MaxRequestsPerChild 100\n";
	    }
	}
	
	# ServerSignature Off, On, or with Email
	elsif ($lines[$i] =~ m/^\s*ServerSignature/) {
	    my $servsig = eval { $apachedoc->getElementsByTagName("ServerSignature")->item(0)->getFirstChild->getData(); } || "";
	    if ($servsig ne "") {
		$lines[$i] = "ServerSignature $servsig\n";
	    }
	}

	# CacheNegotiatedDocuments on or off
	elsif ($lines[$i] =~ m/^#\s*CacheNegotiatedDocs\n/) {
	    my $cachedocs = $apachedoc->getElementsByTagName("CacheNegotiatedDocs")->item(0)->getFirstChild->getData();
	    if ($cachedocs eq "Yes") {
		$lines[$i] =~ s/^#\s*CacheNegotiatedDocs/CacheNegotiatedDocs/;
	    }
	}
	elsif ($lines[$i] =~ m/^\s*CacheNegotiatedDocs\n/) {
	    my $cachedocs = eval { $apachedoc->getElementsByTagName("CacheNegotiatedDocs")->item(0)->getFirstChild->getData(); } || "";
	    if ($cachedocs eq "No") {
		$lines[$i] =~ s/^\s*CacheNegotiatedDocs/# CacheNegotiatedDocs/;
	    }
	}

	# Use Canonical Name
	elsif ($lines[$i] =~ m/^\s*UseCanonicalName/) {
	    my $canonname = eval { $apachedoc->getElementsByTagName("UseCanonicalName")->item(0)->getFirstChild->getData(); } || "";
	    if ($canonname ne "") {
		$lines[$i] = "UseCanonicalName $canonname\n";
	    }
	}
	
	# HostLookUps on or off
	elsif ($lines[$i] =~ m/^\s*HostnameLookups/) {
	    my $hostnameup = eval { $apachedoc->getElementsByTagName("HostnameLookups")->item(0)->getFirstChild->getData(); } || "";
	    if ($hostnameup ne "") {
		$lines[$i] = "HostnameLookups $hostnameup\n";
	    }
	}

	# Index Options (fancy indexing or standard)
	elsif ($lines[$i] =~ m/^\s*IndexOptions/) {
	    my $indexoptions = eval { $apachedoc->getElementsByTagName("IndexOptions")->item(0)->getFirstChild->getData(); } || "";
	    $lines[$i] = "IndexOptions $indexoptions\n";
	}

	# Location of the TypesConfig file (mime.types)
	elsif ($lines[$i] =~ m/^\s*TypesConfig/) {
	    my $typesconf = eval { $apachedoc->getElementsByTagName("TypesConfig")->item(0)->getFirstChild->getData(); } || "";
	    if ($typesconf ne "") {
		$lines[$i] = "TypesConfig $typesconf\n";
	    }
	    else {
		$lines[$i] = "TypesConfig /etc/mime.types\n";
	    }
	}

	# Distinguish the type of default MIME type
	elsif ($lines[$i] =~ m/^\s*DefaultType/) {
	    my $defaulttype = eval { $apachedoc->getElementsByTagName("DefaultType")->item(0)->getFirstChild->getData(); } || "";
	    if ($defaulttype ne "") {
		$lines[$i] = "DefaultType $defaulttype\n";
	    }
	    else {
		$lines[$i] = "DefaultType text/plain\n";
	    }
	}

	# Location of the Error Log
	elsif ($lines[$i] =~m/^\s*ErrorLog/) {
	    my $errorlog = eval { $apachedoc->getElementsByTagName("ErrorLog")->item(0)->getFirstChild->getData(); } || "";
	    if ($errorlog ne "") {
		$lines[$i] = "ErrorLog $errorlog\n";
	    }
	}

	# Type of error message logged 
	elsif ($lines[$i] =~ m/^\s*LogLevel/) {
	    my $loglevel = eval { $apachedoc->getElementsByTagName("LogLevel")->item(0)->getFirstChild->getData(); } || "";
	    if ($loglevel ne "") {
		$lines[$i] = "LogLevel $loglevel\n";
	    }
	}

	# The length of the Timeout
	elsif ($lines[$i] =~ m/^\s*Timeout/) {
	    my $timeout = eval { $apachedoc->getElementsByTagName("Timeout")->item(0)->getFirstChild->getData(); } || "";
	    if ($timeout ne "") {
		$lines[$i] = "Timeout $timeout\n";
	    }
	}

	# Determine if Keep Alive is on or off
	elsif ($lines[$i] =~ m/^\s*KeepAlive /) {
	    my $keepalive = eval { $apachedoc->getElementsByTagName("KeepAlive")->item(0)->getFirstChild->getData(); } || "";
	    if ($keepalive ne "") {
		$lines[$i] = "KeepAlive $keepalive\n";
	    }
	}

	# Determine length of Max. Keep Alive requests
	elsif ($lines[$i] =~ m/^\s*MaxKeepAliveRequests/) {
	    my $maxkareq = eval { $apachedoc->getElementsByTagName("MaxKeepAliveRequests")->item(0)->getFirstChild->getData(); } || "";
	    if ($maxkareq ne "") {
		$lines[$i] = "MaxKeepAliveRequests $maxkareq\n";
	    }
	}
	
	# Determine length of Keep Alive timeouts
	elsif ($lines[$i] =~ m/^\s*KeepAliveTimeout/) {
	    my $katimeout = eval { $apachedoc->getElementsByTagName("KeepAliveTimeout")->item(0)->getFirstChild->getData(); } || "";
	    if ($katimeout ne "") {
		$lines[$i] = "KeepAliveTimeout $katimeout\n";
	    }
	}

	# Determine Language Priority
	elsif ($lines[$i] =~ m/^\s*LanguagePriority/) {
	    my $langprior = eval { $apachedoc->getElementsByTagName("LanguagePriority")->item(0)->getFirstChild->getData(); } || "";
	    if ($langprior ne "") {
		$lines[$i] = "LanguagePriority $langprior\n";
	    }
	}

	
	# IP/Port to listen on
#  	elsif ($lines[$i] =~ m/^\s*Listen/) {
#  	    my $listen = eval { $apachedoc->getElementsByTagName("Listen")->item(0)->getFirstChild->getData(); } || "";
#  	    if ($listen ne "") {
#  		$lines[$i] = "Listen $listen\n";
#  	    }
#  	}
	
#  	# Port to listen on
#  	elsif ($lines[$i] =~ m/^\s*Port/) {
#  	    my $port = eval { $apachedoc->getElementsByTagName("Port")->item(0)->getFirstChild->getData(); } || "80";
#  	    if ($port ne "") {
#  		$lines[$i] = "Port $port\n";
#  	    }
#  	}
	
	# Enable or unenable the SSL engine for the virtual host
	elsif ($lines[$i] =~ m/^\s*SSLEngine/) {

	    my $sslengine = eval { $apachedoc->getElementsByTagName("SSLEngine")->item(0)->getFirstChild->getData(); } || "";

	    general::debug( "Updating SSLEngine settings $sslengine");

	    if ($sslengine ne "") {
		if ($sslengine eq "Yes") {
		    $lines[$i] = "SSLEngine on\n";
		}
		else {
		    $lines[$i] = "SSLEngine off\n";
		}
	    }
	}	
    }

    # Set permissions for root web
    my $serveradmin = eval { $apachedoc->getElementsByTagName("ServerAdmin")->item(0)->getFirstChild->getData(); } || "";

    my %userInfo = datastore::userInfoByID($serveradmin);

    if ($userInfo{'loginname'} eq "") {
	$serveradmin = "root";
    }
    else {
	$serveradmin = $userInfo{'loginname'};
    }

    # Change ownership to new server admin
    my $command = "/bin/chown -R $serveradmin:$serveradmin /var/www";
    my $result = `$command`;

#    general::debug( `/usr/bin/whoami`);
#    general::debug( "Command = $command");

    my $servname = eval { $apachedoc->getElementsByTagName("ServerName")->item(0)->getFirstChild->getData(); } || "";

    # Remove the old sym link
    $command = "/bin/rm /home/$oldServerAdmin/$servname";
  general::debug("Removing old sym link $command");
    $result = `$command`;

    # Build a symbolic link to the root web server
    $command = "/bin/ln -s /var/www/ /home/$userInfo{'loginname'}/$servname";
    $result = `$command`;

    # Install the front page services if they don't exist
    if ((-s "/var/www/html/_vti_pvt/service.pwd") <= 0) {

	# Make sure fphttpd.conf exists
	if (open(FPFILE, ">/etc/httpd/conf/fphttpd.conf")) {

	    my $command = "/bin/cat $system{'httpd.conf'}";
	    my $result = `$command`;

	    print FPFILE "$result\n";

	    $command = "/bin/cat /etc/httpd/conf/vhosts.conf";
	    $result = `$command`;

	    print FPFILE "$result\n";

	    close(FPFILE);

	    $command = "/bin/chmod o+r /etc/httpd/conf/fphttpd.conf";
	    $result = `$command`;

	}

	`/bin/rm /usr/local/frontpage/we80.cnf`;

	$command = "echo \"0\n\n$serveradmin\n$serveradmin\n$serveradmin\ntemporary442\ntemporary442\n\" | /usr/local/frontpage/version4.0/bin/fpsrvadm_suid.exe -operation install -port 80 -servconf /etc/httpd/conf/fphttpd.conf";
	# WARNING:  Do not remove this STDERR!!
	general::debug( "Command for building front page = $command");
	$result = `$command`;

    }


    # Update the root web front page settings
    # Update the front page password
    my $result = `/bin/rm /var/www/html/_vti_pvt/service.pwd`;
    $result = `/bin/rm /var/www/html/_vti_pvt/service.grp`;

    # Update the password files
    # Rebuild the service.grp file
    if (open (FPFILE, ">/var/www/html/_vti_pvt/service.grp")) {
	print FPFILE "# -FrontPage-\n";
	print FPFILE "administrators: $serveradmin\n";
	
	print FPFILE "authors: ";
	
	close(FPFILE);
    }

    my $fpextensions = `/bin/cat $system{'datastore'}/apache.xml | /bin/grep \\\<FrontPageExtensions\\\>Yes`;

    # If front page is on, add it to the root web
    if ($fpextensions ne "") {

	# Make mock module
	`/bin/cp /usr/lib/apache/mod_example.so /usr/lib/apache/mod_frontpage.so`;
	`$system{'root'}/system/bin/chown root:root /usr/lib/apache/mod_example.so /usr/lib/apache/mod_frontpage.so`;
	`$system{'root'}/system/bin/chmod +x /usr/lib/apache/mod_frontpage.so`;
	
	# Rebuild the service.pwd file
	if (open (FPFILE, ">/var/www/html/_vti_pvt/service.pwd")) {
	    print FPFILE "# -FrontPage-\n";
	    close(FPFILE);
	}

    }
    else {

	# Make mock module
	`/bin/rm /usr/lib/apache/mod_frontpage.so`;

    }
    
    updateHTPasswd ($serveradmin, "/var/www/html/_vti_pvt/service.pwd");
    
    # Update .htaccess to play nicely with autoindex
    if (open(HTACCESS, "/var/www/html/.htaccess")) {
	
	my @htlines = <HTACCESS>;
	
	close(HTACCESS);
	
	if (open(HTACCESS, ">/var/www/html/.htaccess")) {
	    
	    foreach my $htline (@htlines) {
		
		if ($htline =~ m/IndexIgnore/) {
		    $htline = "<IfModule mod_autoindex.c>
  IndexIgnore .htaccess */.??* *~ *# */HEADER* */README* */_vti*
</IfModule>\n";
		}
		
		print HTACCESS $htline;
	    }
	    
	    close(HTACCESS);
	}
    }


    $command = "$system{'root'}/system/bin/chmod ug+ws /var/www/html/_vti_bin/shtml.exe";
    $result = `$command`;
    $command = "$system{'root'}/system/bin/chmod ug+ws /var/www/html/_vti_bin/fpcount.exe";
    $result = `$command`;
    $command = "$system{'root'}/system/bin/chmod ug+ws /var/www/html/_vti_bin/_vti_adm/admin.exe";
    $result = `$command`;
    $command = "$system{'root'}/system/bin/chmod ug+ws /var/www/html/_vti_bin/_vti_aut/author.exe";
    $result = `$command`;

    # Open config file
    if (open(FILE, ">$configFile")) {
    
	# Write lines
	foreach my $line (@lines) {
	    print FILE $line;
	}
	
	# Close config file
	close(FILE);

	general::debug( "Wrote httpd.conf back out");

    }
    else {
	general::debug( "WARNING:  Unable to write to httpd.conf");
    }

    # Clean-up
    $apachedoc->dispose();

    # Make sure fphttpd.conf exists
    if (open(FPFILE, ">/etc/httpd/conf/fphttpd.conf")) {
	
	my $command = "/bin/cat $system{'httpd.conf'}";
	my $result = `$command`;
	
	print FPFILE "$result\n";
	
	$command = "/bin/cat /etc/httpd/conf/vhosts.conf";
	$result = `$command`;

	print FPFILE "$result\n";
	
	close(FPFILE);
	
	$command = "/bin/chmod o+r /etc/httpd/conf/fphttpd.conf";
	$result = `$command`;
	
    }

    return(1);

}

# A sub-routine that starts the Apache web server
sub startApache {

    my %system = config::system();

    my $command = "$system{'root'}/system/bin/apache start";
    my $result = `$command`;

    return(1);

}

# A sub-routine that cycles the Apache web server
sub restartApache {

    my %system = config::system();

    my $command = "$system{'root'}/system/bin/apache restart";
    my $result = `$command`;

    # Check to make sure we are running 
    my $processes = `/bin/ps auwx | /bin/grep /usr/sbin/httpd | /usr/bin/wc -l`;
    $processes =~ s/\W//g;
    $processes = scalar($processes);

    general::debug( "Found $processes httpd daemons already running");

    if ($processes < 5) {

	# Remove any lock files
	my $command = "/bin/rm -f /var/lock/subsys/httpd /var/run/httpd.pid";
	my $result = `$command`;

	stopApache();
	startApache();

	general::debug( "Apache manual restart!");

    }

    general::debug( "Completely finished with apache restart!");

    return(1);


}

# A sub-routine that adds a virtual site to the system
sub addVirtualSite {

    my %system = config::system;

    my $sitename = shift || return(0);
    my $diskquota = shift || "0";
    my @members = @_;
    
    # First grab all IP addresses

    my $parser = new XML::DOM::Parser;
    my $doc = $parser->parsefile("$system{'datastore'}/network.xml");

    my $interfaces = $doc->getElementsByTagName("Interfaces")->item(0)->getElementsByTagName("Interface",0);
    my $numOfInterfaces = $interfaces->getLength;

    my $array = "";

    for (my $j=0; $j<$numOfInterfaces;$j++) {

	my $interfaceNode = $interfaces->item($j);
	my $ip1 = eval {$interfaceNode->getElementsByTagName("IP")->item(0)->getFirstChild->getData();} || "";

	$array .= "$ip1<TD>";
    }
    
    # Split out the values
    my @ips = split(/<TD>/,$array);
    my $iplist = join(' ', @ips);

    # If the vhosts file does not exists, populate it with the root web
    if  ((-s $system{'vhostsfile'}) <= 0) {

	`/bin/rm /usr/local/frontpage/we80.cnf`;

	open(VHOSTFILE, ">$system{'vhostsfile'}");

	print VHOSTFILE "<VirtualHost $iplist>
   DocumentRoot \"/var/www/html\"\n";

	my $fpextensions = `/bin/cat $system{'datastore'}/apache.xml | /bin/grep \\\<FrontPageExtensions\\\>Yes`;

	# If front page is on, add it to the root web
	if ($fpextensions ne "") {

	    print VHOSTFILE "

  <Location \"/_vti_pvt\">
    Order deny,allow
    Deny from all
  </Location>
  <Location \"/_vti_cnf\">
    Order deny,allow
    Deny from all
  </Location>

<IfDefine HAVE_FRONTPAGE>
  ScriptAlias /_vti_bin/_vti_adm/ /var/www/html/_vti_b\in/_vti_adm/
  ScriptAlias /_vti_bin/_vti_aut/ /var/www/html/_vti_b\in/_vti_aut/
  ScriptAlias /_vti_bin/ /var/www/html/_vti_bin/
</IfDefine>

";
	    
	}


	print VHOSTFILE "</VirtualHost>\n";

	foreach my $ip (@ips) {
	    print VHOSTFILE "NameVirtualHost $ip\n";
	}

	close(VHOSTFILE);
    }
    
    # Read the information for this virtual host
    my $parser1 = new XML::DOM::Parser;
    my $doc1 = $parser1->parsefile("$system{'datastore'}/virtualsites.xml");

    my $virtualSites = eval { $doc1->getElementsByTagName("VirtualHost"); } || return(0);
    my $numOfVirtualSites = $virtualSites->getLength();

    # Find the right site
    my $siteNode = "";
    for (my $i=0; $i < $numOfVirtualSites; $i++) {

	my $thisSite = $virtualSites->item($i)->getElementsByTagName("ServerName")->item(0)->getFirstChild->getData();

	if ($thisSite eq $sitename) { 
	    $siteNode = $virtualSites->item($i);
	}
    }

    if ($siteNode eq "") {
	return(0);
    }
    else {

	# Parse information about log rotation from XML
	my $logparser = new XML::DOM::Parser;
	my $logdoc = $parser->parsefile("$system{'datastore'}/general.xml");

	# Grab the values for the log files
	my $weblog = eval { $logdoc->getElementsByTagName("Logs")->item(0)->getElementsByTagName("Weblog")->item(0)->getFirstChild->getData(); } || "";

	my $weblogfile = "
/var/log/httpd/access_log {
    missingok
    postrotate
        /bin/kill -HUP `cat /var/run/httpd.pid 2>/dev/null` 2> /dev/null || true
    endscript
}

/var/log/httpd/agent_log {
    missingok
    postrotate
        /bin/kill -HUP `cat /var/run/httpd.pid 2>/dev/null` 2> /dev/null || true
    endscript
}

/var/log/httpd/error_log {
    missingok
    postrotate
        /bin/kill -HUP `cat /var/run/httpd.pid 2>/dev/null` 2> /dev/null || true
    endscript
}

/var/log/httpd/referer_log {
    missingok
    postrotate
        /bin/kill -HUP `cat /var/run/httpd.pid 2>/dev/null` 2> /dev/null || true
    endscript
}
";

	# weblog
	if ($weblog eq "Once a Month") {
	    $weblogfile =~ s/missingok/missingok\n     monthly/g;
	}
	elsif ($weblog eq "When greater than 5 MB") {
	    $weblogfile =~ s/missingok/missingok\n     size=5M/g;
	}
	elsif ($weblog eq "When greater than 10 MB") {
	    $weblogfile =~ s/missingok/missingok\n     size=10M/g;
	}
	elsif ($weblog eq "When greater than 50 MB") {
	    $weblogfile =~ s/missingok/missingok\n     size=50M/g;
	}
	else {
	    $weblogfile =~ s/missingok/missingok\n     weekly/g;
	}
	
	# Grab information about the virtual site
	my $docroot = $siteNode->getElementsByTagName("DocumentRoot")->item(0)->getFirstChild->getData(); 
	my $siteadmin = $siteNode->getElementsByTagName("ServerAdmin")->item(0)->getFirstChild->getData();

	my $frontpage = eval { $siteNode->getElementsByTagName("AllowFrontPage")->item(0)->getFirstChild->getData(); } || "";
	
	my $servername = eval { $siteNode->getElementsByTagName("ServerName")->item(0)->getFirstChild->getData(); } || "";
	my $group = eval { $siteNode->getElementsByTagName("Group")->item(0)->getFirstChild->getData(); } || "";
	my $status = eval { $siteNode->getElementsByTagName("Status")->item(0)->getFirstChild->getData(); } || "";
	my $cgi = eval { $siteNode->getElementsByTagName("AllowCGI")->item(0)->getFirstChild->getData(); } || "";
	my $frontpage = eval { $siteNode->getElementsByTagName("AllowFrontPage")->item(0)->getFirstChild->getData(); } || "";
	my $php = eval { $siteNode->getElementsByTagName("AllowPHP")->item(0)->getFirstChild->getData(); } || "";
	my $modperl = eval { $siteNode->getElementsByTagName("AllowModPerl")->item(0)->getFirstChild->getData(); } || "";
	my $bandwidth = eval { $siteNode->getElementsByTagName("QOS")->item(0)->getElementsByTagName("Bandwidth")->item(0)->getFirstChild->getData(); } || "";
	my $limit_cpu = eval { $siteNode->getElementsByTagName("Limits")->item(0)->getElementsByTagName("CPU")->item(0)->getFirstChild->getData(); } || "";
	my $limit_memory = eval { $siteNode->getElementsByTagName("Limits")->item(0)->getElementsByTagName("Memory")->item(0)->getFirstChild->getData(); } || "";
	my $limit_procs = eval { $siteNode->getElementsByTagName("Limits")->item(0)->getElementsByTagName("Processes")->item(0)->getFirstChild->getData(); } || "";

	my $ip = eval { $siteNode->getAttribute("ip"); } || "";
	if ($ip eq "" || $ip eq "default") { $ip = "$iplist"; }

	# Grab some additional user information
	my %userInfo = datastore::userInfoByID($siteadmin);

	my $adminemail = "$userInfo{'loginname'}\@$servername";

	# Add the Virtual Site group
	my $groupID = addVSGroup($sitename, $sitename, $diskquota, @members);

	# Calculate primary directory
	my $directory = $docroot;
	$directory =~ s/\/html//;

	# Make symbolic links to user home directories
	foreach my $member (@members) {

	    my %memberInfo = datastore::userInfoByID($member);

	    # Make sym link for site
	    if ($memberInfo{'loginname'} ne "") {
		my $command = "/bin/ln -s $directory /home/$memberInfo{'loginname'}/$servername";
		my $result = `$command`;
		$command = "/bin/chown $memberInfo{'loginname'}:$memberInfo{'loginname'} /home/$memberInfo{'loginname'}/$servername";
		$result = `$command`;
		general::debug( "Result from sym link = $result");
	    }
	}

	my $command = "/bin/ln -s $directory /home/$userInfo{'loginname'}/$servername";
	my $result = `$command`;
	$command = "/bin/chown $userInfo{'loginname'}:$userInfo{'loginname'} /home/$userInfo{'loginname'}/$servername";
	$result = `$command`;
	
	# Update group node
      datastore::textNodeUpdate($siteNode->getElementsByTagName("Group"), $groupID);

	# Write the file back out
      datastore::XMLtoFile($doc1,"$system{'datastore'}/virtualsites.xml");

	# Add site administrator to list of site_admins
	users::addUserToOSGroup($userInfo{'loginname'}, "site_admins");

	# Make primary directory
	$command = "/bin/mkdir $directory";
	$result = `$command`;

	# Make the document root directory
	$command = "/bin/mkdir $docroot";
	$result = `$command`;

	# Make the cgi-bin directory
	$command = "/bin/mkdir $directory/cgi-bin";
	$result = `$command`;

	# Make the mod_perl directory
	$command = "/bin/mkdir $directory/mod_perl";
	$result = `$command`;

	# Make the log directory
	$command = "/bin/mkdir $directory/logs";
	$result = `$command`;

	# Humor webalizer and create 5 access log files
	$command = "/bin/touch $directory/logs/access_log";
	$result = `$command`;
	$command = "/bin/chown root:root $directory/logs/error_log";
	$result = `$command`;
	$command = "/bin/chown root:root $directory/logs/access_log";
	$result = `$command`;

	# Make log rotation file for this virtual site
	$weblogfile =~ s/\/var\/log\/httpd\//$directory\/logs\//g;
	if (open(ROTATEFILE, ">/etc/logrotate.d/$sitename")) {

	    print ROTATEFILE $weblogfile;

	    close (ROTATEFILE);

	}

	# Setup webalizer
	if (open (FILE, ">$system{'root'}/system/reports/webalizer_$sitename.conf")) {

	    $command = "/bin/mkdir $directory/html/webalizer";
	    $result = `$command`;

	    print FILE "LogType clf\n";

	    print FILE "OutputDir     $directory/html/webalizer\n";

	    print FILE "HistoryName    webalizer.hist\n";

	    print FILE "HostName        $sitename\n\n";

	    print FILE "PageType	htm*\n";
	    print FILE "PageType	cgi\n";
	    print FILE "PageType	phtml\n";
	    print FILE "PageType	php3\n";
	    print FILE "PageType	pl\n\n";

	    print FILE "UseHTTPS        Yes\n";

	    print FILE "HTMLPre <!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\">\n";
	    print FILE "HTMLHead <META NAME=\"Webalizer\" CONTENT=\"The Webalizer\">\n";
	    print FILE "HTMLBody <BODY BGCOLOR=\"#E8E8E8\" TEXT=\"#000000\" LINK=\"#0000FF\" VLINK=\"#FF0000\">\n";
	    print FILE "HTMLPost 	<BR CLEAR=\"all\">\n";
	    print FILE "HTMLEnd </BODY></HTML>\n";

	    print FILE "Quiet		no\n";
	    print FILE "ReallyQuiet	no\n";

	    print FILE "TimeMe		no\n";

	    print FILE "GMTTime		no\n";

	    print FILE "Debug		no\n";

	    print FILE "FoldSeqErr	no\n";

	    print FILE "VisitTimeout	1800\n";

	    print FILE "IgnoreHist	no\n";

	    print FILE "CountryGraph	yes\n";

	    print FILE "DailyGraph	yes\n";
	    print FILE "DailyStats	yes\n";

	    print FILE "HourlyGraph	yes\n";
	    print FILE "HourlyStats	yes\n";

	    print FILE "GraphLegend	yes\n";

	    print FILE "GraphLines	2\n";

	    print FILE "TopSites        30\n";
	    print FILE "TopKSites       10\n";
	    print FILE "TopURLs         30\n";
	    print FILE "TopKURLs        10\n";
	    print FILE "TopReferrers    30\n";
	    print FILE "TopAgents       15\n";
	    print FILE "TopCountries    30\n";
	    print FILE "TopEntry        10\n";
	    print FILE "TopExit         10\n";
	    print FILE "TopSearch       20\n";
	    print FILE "TopUsers        20\n";

	    print FILE "AllSites	no\n";
	    print FILE "AllURLs	        no\n";
	    print FILE "AllReferrers	no\n";
	    print FILE "AllAgents	no\n";
	    print FILE "AllSearchStr	no\n";
	    print FILE "AllUsers        no\n";

	    print FILE "IndexAlias     home.htm\n";
	    print FILE "IndexAlias	homepage.htm\n";

	    print FILE "HideURL		*.gif\n";
	    print FILE "HideURL		*.GIF\n";
	    print FILE "HideURL		*.jpg\n";
	    print FILE "HideURL		*.JPG\n";
	    print FILE "HideURL		*.png\n";
	    print FILE "HideURL		*.PNG\n";
	    print FILE "HideURL		*.ra\n";

	    print FILE "HideAllSites	no\n";

	    print FILE "GroupDomains	0\n";

	    print FILE "GroupShading	        yes\n";
	    print FILE "GroupHighlight	yes\n";

	    print FILE "MangleAgents    0\n";

	    print FILE "SearchEngine	yahoo.com	p=\n";
	    print FILE "SearchEngine	altavista.com	q=\n";
	    print FILE "SearchEngine	google.com	q=\n";
	    print FILE "SearchEngine	eureka.com	q=\n";
	    print FILE "SearchEngine	lycos.com	query=\n";
	    print FILE "SearchEngine	hotbot.com	MT=\n";
            print FILE "SearchEngine	msn.com		MT=\n";
	    print FILE "SearchEngine	infoseek.com	qt=\n";
	    print FILE "SearchEngine	webcrawler	searchText=\n";
	    print FILE "SearchEngine	excite		search=\n";
	    print FILE "SearchEngine	netscape.com	search=\n";
	    print FILE "SearchEngine	mamma.com	query=\n";
	    print FILE "SearchEngine	alltheweb.com	query=\n";
	    print FILE "SearchEngine	northernlight.com  qr=\n";

	    print FILE "DumpHeader	no\n";
	    print FILE "DumpExtension	tab\n";
	    print FILE "DumpSites	no\n";
	    print FILE "DumpURLs	no\n";
	    print FILE "DumpReferrers	no\n";
	    print FILE "DumpAgents	no\n";
	    print FILE "DumpUsers	no\n";
	    print FILE "DumpSearchStr  no\n";
		     
	    close(FILE);

	    # Change ownership permissions
	    $command = "/bin/chown -R $siteadmin:$groupID $docroot/webalizer";
	    $result = `$command`;

	}

	# Place a temporary index file in the site directory
	if (open (FILE, ">$docroot/index.html")) {

	    print FILE "<HTML>
           <BODY bgcolor=white>
              <H2><CENTER>This is the future site for \"$sitename\".</CENTER></H2>
           </BODY>
           </HTML>";

	    close (FILE);
	}

	# Make a place holder for the webalizer report
	open (FILE, ">$docroot/webalizer/index.html");
	print FILE "<HTML>
           <BODY bgcolor=white>
              <H2><CENTER>Web server statistics have not yet run for \"$sitename\".  When they have run, this report will graphically illustrate your site usage.</CENTER></H2>
           </BODY>
           </HTML>";
	close (FILE);

	# Add the Virtual Site to the vhosts.conf file
	if (open(VHOSTFILE, "<$system{'vhostsfile'}")) {

	    my @lines = <VHOSTFILE>;

	    close(VHOSTFILE);

	    if (open(VHOSTFILE, ">$system{'vhostsfile'}")) {

		# Print all existing configurations
		foreach my $line (@lines) {
		    print VHOSTFILE $line;
		}

		# Add the new one if active
		if ($status eq "Active") {
		
		    print VHOSTFILE "
<VirtualHost $ip>
  ServerName $servername
  ServerAdmin $adminemail
  CustomLog $directory/logs/access_log common
  LogLevel warn
  ErrorLog $directory/logs/error_log
  DocumentRoot $docroot

  <Directory $docroot>
    AllowOverride All
  </Directory>
";

		    # Setup basic cgi
		    if ($cgi eq "Yes") {

			print VHOSTFILE "
  ScriptAlias /cgi-bin/ \"$directory/cgi-bin/\"

  <Directory \"$directory/cgi-bin\">
    AllowOverride None
    Options ExecCGI
    Order allow,deny
    Allow from all
  </Directory>

";
		    }

		    # Deny access to php served pages
		    if ($php ne "Yes") {
			print VHOSTFILE "<Files ~ \"\\.*(php3|php4|php|phtml)\">
   Order allow,deny
   Deny from all
</Files>

";
		    }


		    # Enabling/Disabling mod_perl
		    if ($modperl eq "Yes") {

			# Setup mod_perl
			print VHOSTFILE "
<IfModule mod_perl.c>

  ScriptAlias /mod_perl/ \"$directory/mod_perl/\"

  <Location /mod_perl>
    SetHandler perl-script
    PerlHandler Apache::Registry
    Options +ExecCGI
  </Location>
</IfModule>

";

		}

		    # Setup mod_throttle
		    # if ($bandwidth ne "" && $bandwidth ne "0") {
		    # print VHOSTFILE "
# ThrottlePolicy Volume $bandwidth 1s\n";
#		    }

		    # Setup DoS Limits
		    if ($limit_cpu ne "") {
			print VHOSTFILE "
  RLimitCPU     $limit_cpu";
		    }
		    if ($limit_procs ne "") {
			print VHOSTFILE "
  RLimitNPROC     $limit_procs";
		    }
		    if ($limit_memory ne "") {
			print VHOSTFILE "
  RLimitMEM     $limit_memory";
		    }
		    
		    # Setup Front Page Extensions
		    print VHOSTFILE "
  <Location \"/_vti_pvt\">
    Order deny,allow
    Deny from all
  </Location>
  <Location \"/_vti_cnf\">
    Order deny,allow
    Deny from all
  </Location>
";

		    if ($frontpage eq "Yes") {

			print VHOSTFILE "
<IfDefine HAVE_FRONTPAGE>
  ScriptAlias /_vti_bin/_vti_adm/ /home/VirtualSites/$servername/html/_vti_b\in/_vti_adm/
  ScriptAlias /_vti_bin/_vti_aut/ /home/VirtualSites/$servername/html/_vti_b\in/_vti_aut/
  ScriptAlias /_vti_bin/ /home/VirtualSites/$servername/html/_vti_bin/
</IfDefine>
";

		    }
		    else {
			print VHOSTFILE "
  <Location \"/_vti_bin\">
    Order deny,allow
    Deny from all
  </Location>
";
		    }

		    print VHOSTFILE "
</VirtualHost>\n";


		    # Update permissions on the directories
		    my $command = "/bin/chown $siteadmin:$group $docroot";
		    my $result = `$command`;

		}

	    }

	    close(VHOSTFILE);
	}

	my $dirs = `/bin/ls $directory/html/*`;
	general::debug( "Before running fpsrvadmin $dirs");

	# Make Front Page happy
	if (open(FPFILE, ">/etc/httpd/conf/fphttpd.conf")) {

	    my $command = "/bin/cat $system{'httpd.conf'}";
	    my $result = `$command`;

	    print FPFILE "$result\n";

	    $command = "/bin/cat /etc/httpd/conf/vhosts.conf";
	    $result = `$command`;

	    print FPFILE "$result\n";

	    close(FPFILE);

	    $command = "/bin/chmod o+r /etc/httpd/conf/fphttpd.conf";
	    $result = `$command`;

	}

	general::debug( "Building Front Page Directories");

	# Make FP directories
	$command = "echo \"0\n$userInfo{'loginname'}\n$groupID\n$userInfo{'loginname'}\ntemporary442\ntemporary442\n\" | /usr/local/frontpage/version4.0/bin/fpsrvadm_suid.exe -operation install -port 80 -multihost $sitename -servconf /etc/httpd/conf/fphttpd.conf";
	# WARNING:  Do not remove this STDERR!!
	general::debug( "Command for building front page = $command");
	$result = `$command`;
	
	general::debug( "FRONTPAGE created with $command\n\n Results = $result");

	my $result = `/bin/rm $docroot/_vti_pvt/service.pwd`;
	$result = `/bin/rm $docroot/_vti_pvt/service.grp`;

	# Update the password files
	# Rebuild the service.grp file
	if (open (FPFILE, ">$docroot/_vti_pvt/service.grp")) {
	    print FPFILE "# -FrontPage-\n";
	    print FPFILE "administrators: $userInfo{'loginname'}\n";
	    
	    print FPFILE "authors: ";
	    
	    close(FPFILE);
	}
	
	# Rebuild the service.pwd file
	if (open (FPFILE, ">$docroot/_vti_pvt/service.pwd")) {
	    print FPFILE "# -FrontPage-\n";
	    close(FPFILE);
	}
	
	updateHTPasswd ($userInfo{'loginname'}, "$docroot/_vti_pvt/service.pwd");

	# Update .htaccess to play nicely with autoindex
	if (open(HTACCESS, "$docroot/.htaccess")) {

	    my @htlines = <HTACCESS>;

	    close(HTACCESS);

	    if (open(HTACCESS, ">$docroot/.htaccess")) {

		foreach my $htline (@htlines) {

		    if ($htline =~ m/IndexIgnore/) {
			$htline = "<IfModule mod_autoindex.c>
  IndexIgnore .htaccess */.??* *~ *# */HEADER* */README* */_vti*
</IfModule>\n";
		    }

		    print HTACCESS $htline;
		}

		close(HTACCESS);
	    }
	}
	
	general::debug( "Changing ownership");

	# Change ownership permissions
	$command = "/bin/chown $siteadmin:$groupID $docroot/index.html";
	$result = `$command`;

	$command = "/bin/chown -R $siteadmin:$groupID $directory/html/_vti_pvt/service.pwd";
	$result = `$command`;	
	$command = "/bin/chown -R $siteadmin:$groupID $directory/html/_vti_pvt/service.grp";
	$result = `$command`;	

	# Fix permissions on lock file
	$command = "/bin/chmod o-wx $directory/html/_vti_pvt/frontpg.lck";
	$result = `$command`;	

	$command = "/bin/chown -R $siteadmin:$groupID $directory";
	$result = `$command`;
	$command = "/bin/chown -R apache:$groupID $directory";
	$result = `$command`;
	$command = "/bin/chmod g+ws $directory";
	$result = `$command`;
	$command = "/bin/chmod o-rwx $directory";
	$result = `$command`;
	$command = "/bin/chown -R $siteadmin:$groupID $directory/html";
	$result = `$command`;
	$command = "/bin/chmod g+ws $directory/html";
	$result = `$command`;
	$command = "/bin/chown -R $siteadmin:$groupID $directory/logs";
	$result = `$command`;
	$command = "/bin/chmod o-rwx $directory/logs";
	$result = `$command`;
	$command = "/bin/chown -R $siteadmin:$groupID $directory/cgi-bin";
	$result = `$command`;
	$command = "/bin/chmod g+ws $directory/cgi-bin";
	$result = `$command`;
	$command = "/bin/chown -R $siteadmin:$groupID $directory/mod_perl";
	$result = `$command`;
	$command = "/bin/chmod g+ws $directory/mod_perl";
	$result = `$command`;


	$command = "$system{'root'}/system/bin/chmod ug+ws $docroot/_vti_bin/shtml.exe";
	$result = `$command`;
	$command = "$system{'root'}/system/bin/chmod ug+ws $docroot/_vti_bin/fpcount.exe";
	$result = `$command`;
	$command = "$system{'root'}/system/bin/chmod ug+ws $docroot/_vti_bin/_vti_adm/admin.exe";
	$result = `$command`;
	$command = "$system{'root'}/system/bin/chmod ug+ws $docroot/_vti_bin/_vti_aut/author.exe";
	$result = `$command`;

	# Update the sendmail generics domain
	sendmail::updateSendmailGenericsDomain();

	general::debug( "Ownership updated");

	print STDER "Calling update sendmail\n";
	sendmail::updateSendmail();

	# Restart Apache
	stopApache();
	restartApache();

    }

    return(1);
}

# Update the current key files
sub updateApacheSSLKeys {

    my $sslkey = shift || return();
    my $sslcsr = shift || return();
    my $sslcrt = shift || return();
    
    general::debug( "About to update SSL files");
    
    # Update server key
    if (open (SSLFILE, ">/etc/httpd/conf/ssl.key/server.key")) {
	
	print SSLFILE "$sslkey";
	
	close (SSLFILE);

	general::debug( "Wrote out server key");
    }
    else {
	general::debug( "Could not open server.key for update");
    }
    
    # Update Certificate Request
    if (open (SSLFILE, ">/etc/httpd/conf/ssl.csr/server.csr")) {
	
	print SSLFILE "$sslcsr";
	
	close (SSLFILE);
	
       general::debug( "Wrote out certificate request");
    }
    else {
	general::debug( "Could not open server.csr for update");
    }
    
    # Update Certificate
    if (open (SSLFILE, ">/etc/httpd/conf/ssl.crt/server.crt")) {
	
	print SSLFILE "$sslcrt";
	
	close (SSLFILE);
	
	general::debug( "Wrote out certificate");
    }
    else {
	general::debug( "Could not open server.crt for update");
    }

    return (1);

}


# A sub-routine to re-build the server key, csr, and crt
sub apacheSSLKeyRegen {

    my $passphrase = shift || "";
    my $country = shift || "";
    my $state = shift || "";
    my $city = shift || "";
    my $orgname = shift || "";
    my $orgunit = shift || "";
    my $cname = shift || "";
    my $email = shift || "";
    my $challenge = shift || "";
    my $coname = shift || "";

    my %system = config::system();

    my $keyresponse = "$passphrase\n$passphrase\n";
    my $csrresponse = "$passphrase\n$country\n$state\n$city\n$orgname\n$orgunit\n$cname\n$email\n$challenge\n$coname\n";
    my $crtresponse = "$passphrase\n$country\n$state\n$city\n$orgname\n$orgunit\n$cname\n$email\n";

    # Step 0:  Read in some randomness
    my $rand = "";
    if (open (RANDFILE, "/dev/urandom")) {

	read (RANDFILE, $rand, 4096);

	close (RANDFILE);
    }

    `/bin/echo \"$rand\" > /tmp/rand.txt`;

    # Step 1:  Build new key
    my $command = "/bin/echo \"$keyresponse\" | /usr/bin/openssl genrsa -des3 -rand /tmp/rand.txt 1024 2> /dev/null";
    my $sslkey = `$command`;

    if (open(FILE, ">/etc/httpd/conf/ssl.key/server.key")) {
	print FILE "$sslkey";
	close (FILE);
    }

    `/bin/echo "$sslkey" > $system{'root'}/system/server.key`;

    general::debug( "Finished key");

    # Step 2:  Build new certificate request
    $command = "/bin/echo \"$csrresponse\" | /usr/bin/openssl req -new -key $system{'root'}/system/server.key 2> /dev/null";
    my $sslcsr = `$command`;

    if (open(FILE, ">/etc/httpd/conf/ssl.csr/server.csr")) {
	print FILE "$sslcsr";
	close (FILE);
    }

    general::debug( "Finished certificate request");


      # Step 3:  Remove the pass phrase (It makes the httpd server pause for a pass phrase)
      $command = "echo \"$passphrase\n\" | /usr/bin/openssl rsa -in $system{'root'}/system/server.key 2> /dev/null";
      my $sslcrt_nopass = `$command`;

      if (open(FILE, ">/etc/httpd/conf/ssl.key/server.key")) {

  	print FILE "$sslcrt_nopass";

  	close (FILE);
      }

    `/bin/echo \"$sslcrt_nopass\" > $system{'root'}/system/server.key`;

    general::debug( "Finished with removal of passphrase!");

    # Step 4:  Build temporary certificate
    $command = "/bin/echo \"$crtresponse\" | /usr/bin/openssl req -new -key $system{'root'}/system/server.key -x509 -days 365 2> /dev/null";
    my $sslcrt = `$command`;

    if (open(SSLFILE, ">/etc/httpd/conf/ssl.crt/server.crt")) {

	print SSLFILE "$sslcrt";

	close (SSLFILE);

    }
    else {
	general::debug( "WARNING:  DID NOT WRITE CRT TO FILE!!!");
    }

    general::debug( "Finished certificate");

    # Clean-up
    `/bin/rm $system{'root'}/system/server.key`;
    `/bin/rm $system{'root'}/system/server.crt`;
    `/bin/rm /tmp/rand.txt`;

    # Step 5:  Hard-Cycle server
    $command = `/usr/bin/killall /usr/sbin/httpd`;
    restartApache();

}

# Get the date/time of the SSL files
#  Return key~csr~crt
sub getApacheSSLdates {

    my $result = `/bin/ls -l --full-time /etc/httpd/conf/ssl.key/server.key`;
    my @values = split(' ', $result);
    my $return = "$values[5] $values[6] $values[7] $values[8] $values[9] ";

    $result = `/bin/ls -l --full-time /etc/httpd/conf/ssl.csr/server.csr`;
    @values = split(' ', $result);
    $return .= "~$values[5] $values[6] $values[7] $values[8] $values[9] ";

    $result = `/bin/ls -l --full-time /etc/httpd/conf/ssl.crt/server.crt`;
    @values = split(' ', $result);
    $return .= "~$values[5] $values[6] $values[7] $values[8] $values[9] ";

    return ($return);

}

# Grab the current Apache SSL Key
sub getApacheSSLKey {

    my $sslkey = "";

    if (open (SSLFILE, "</etc/httpd/conf/ssl.key/server.key")) {

	my @sslkeylines = <SSLFILE>;

	foreach my $line (@sslkeylines) {
	    $sslkey .= "$line";
	}

	close (SSLFILE);
    }
    else {
	general::debug( "Could not open server.key");
    }

    return ($sslkey);

}

# Grab the current Apache CSR
sub getApacheSSLCSR {

    my $sslkey = "";

    if (open (SSLFILE, "</etc/httpd/conf/ssl.csr/server.csr")) {

	my @sslkeylines = <SSLFILE>;

	foreach my $line (@sslkeylines) {
	    $sslkey .= "$line";
	}

	close (SSLFILE);
    }
    else {
	general::debug( "Could not open server.csr");
    }

    return ($sslkey);

}

# Grab the current Apache CRT
sub getApacheSSLCRT {

    my $sslkey = "";

    if (open (SSLFILE, "</etc/httpd/conf/ssl.crt/server.crt")) {

	my @sslkeylines = <SSLFILE>;

	foreach my $line (@sslkeylines) {
	    $sslkey .= "$line";
	}

	close (SSLFILE);
    }
    else {
	general::debug( "Could not open server.crt");
    }

    return ($sslkey);

}

# a subroutine used to add the sym link for a particular user to a particular site
sub vs_sym_link {

    my $user = shift || return();
    my $site = shift || return();

    my $command = "/bin/ln -s /home/VirtualSites/$site /home/$user/$site";
    my $result = `$command`;
}

# subroutine used to delete the sym link for a particular user who has been taken from the site
sub del_sym_link {

    my $user = shift || return();
    my $site = shift || return();
    
    general::debug("User to remove sym link from $user for site $site");

    my $command = "/bin/rm /home/$user/$site";
    my $result = `$command`;


}

# A sub-routine to add a group to the system
sub addVSGroup {

    my $shortname = shift || return("");
    my $name = shift || "";
    my $diskquota = shift || "0";
    my @members = @_;

    general::debug( "Adding Virtual Site Group\nMEMBERS = @members");

    my $groupID = "";

    my %system = config::system();

    # Build a temporay group name
    srand();
    my $tempGroupName = "tempgroup" . int(rand(5000));

    # Add the Temporary Group
    my $command = "/usr/sbin/groupadd $tempGroupName";
    my $result = `$command`;
    
    # Grab the group ID
    $result = `/bin/cat /etc/group | /bin/grep ^$tempGroupName:`;
    my @row = split(/:/, $result);
    $groupID = $row[2];

    $shortname = "virtualsite_$groupID";
    
    # Change the Temporary Group to the Real Group
    my @lines = ();
    if (open(FILE, "/etc/group")) {
	
	@lines = <FILE>;
	
	# Replace the group name
	foreach my $line (@lines) {
	    if ($line =~ m/$tempGroupName:/) {
		$line =~ s/$tempGroupName:/$shortname:/;
		last;
	    }
	}
	
	close(FILE);
	
	if (open(FILE, ">/etc/group")) {
	    
	    # Write the file back out
	    foreach my $line (@lines) {
		print FILE "$line";
	    }
	    
	    close(FILE);
	    
	}
	
	# Update the shadow file
	$result = `/usr/bin/grpconv`;
	
	# Update the XML
	my $parser = new XML::DOM::Parser;
	my $doc = $parser->parsefile("$system{'datastore'}/groups.xml");
	
	general::debug( "\n$groupID $shortname $name Members = @members");
	
      datastore::addGroup($doc, $groupID, $shortname, $name, $diskquota, @members);
	
      datastore::XMLtoFile($doc,"$system{'datastore'}/groups.xml");
	
	$doc->dispose();

	groups::updateGroup($shortname);

    }
    
    return($groupID);
    
}
# A sub-routine that stops the Apache web server
sub stopApache {

    my %system = config::system();

    my $command = "$system{'root'}/system/bin/apache stop";
    my $result = `$command`;

    return(1);
}

# A sub-routine that starts the Apache web server
sub startApache {

    my %system = config::system();

    my $command = "$system{'root'}/system/bin/apache start";
    my $result = `$command`;

    return(1);

}

# A sub-routine that starts the Apache daemon
sub restartApache {

    my %system = config::system();

    my $command = "$system{'root'}/system/bin/apache restart";
    my $result = `$command`;

    return(1);

}

# A sub-routine that upgrades the front page extension sites
sub updateVirtualSitesFPConf {

    my %system = config::system();

    # Make Front Page happy
    if (open(FPFILE, ">/etc/httpd/conf/fphttpd.conf")) {

	my $command = "/bin/cat $system{'httpd.conf'}";
	my $result = `$command`;
	
	print FPFILE "$result\n";
	
	$command = "/bin/cat /etc/httpd/conf/vhosts.conf";
	$result = `$command`;
	
	print FPFILE "$result\n";
	
	close(FPFILE);
	
	$command = "/bin/chmod o+r /etc/httpd/conf/fphttpd.conf";
	$result = `$command`;
	
    }

    my $filelist = `/bin/ls -a /usr/local/frontpage/*.cnf`;
    
    my @files = split("\n", $filelist);
    
    my @lines = ();
    
    foreach my $file (@files) {
	
	my $sitename = $file;
	
	$sitename =~ s/\/usr\/local\/frontpage\///;
	my @values = split(":", $sitename);
	$sitename = $values[0];
	
      custom::updateVirtualSite($sitename);
	
	# Read in the existing config file
	if (open(FPFILE, "<$file")) {
	    
	@lines = <FPFILE>;
	
	close(FPFILE);
    }

	if (open(FPFILE, ">$file")) {
	    
	    foreach my $line (@lines) {
		
		if ($line =~ m/serverconfig:/) {
		    $line = "serverconfig:/etc/httpd/conf/fphttpd.conf\n";
		}
		
		print FPFILE "$line";
		
	    }
	    
	    close(FPFILE);
	}
	
    }
    
}

# A sub-routine to update the password in a .htpasswd file
# This sub-routine pulls information from the existing shadow
# password file and populates/adds the .htpasswd file
sub updateHTPasswd {

    my $user = shift || return();
    my $file = shift || return();

    # Open the shadow password file
    my $password = "";
    if (open(FILE, "/etc/shadow")) {

	# Find the correct user
	my @lines = <FILE>;

	foreach my $line (@lines) {
	    if ($line =~ m/^$user:([^:]*):/) {
		$password = $1;
	    }
	}

	close (FILE);

    }

    # Find the user
    if (open(FILE, "<$file")) {

	my $foundUser = 0;
	my @lines = <FILE>;

	foreach my $line (@lines) {

	    if ($line =~ m/$user:/) {
		$foundUser = 1;
		$line = "$user:$password\n";
	    }
	}

	if ($foundUser == 0) {
	    push (@lines, "$user:$password\n");
	}

	# Write the file
	if (open(FILE, ">$file")) {

	    foreach my $line (@lines) {
		print FILE "$line";
	    }

	    close (FILE);
	}
	
    }
    
}

# A sub-routine to remove a user from a .htpasswd file
sub delHTPasswd {

    my $user = shift || return();
    my $file = shift || return();

    # Find the user
    if (open(FILE, "<$file")) {

	my $foundUser = 0;
	my @lines = <FILE>;

	foreach my $line (@lines) {

	    if ($line =~ m/$user:/) {
		$foundUser = 1;
	    }
	}

	if ($foundUser) {

	    # Write the file
	    if (open(FILE, ">$file")) {
		
		foreach my $line (@lines) {

		    # Print every line but the delete user
		    if (!($line =~ m/^$user:/)) {
			print FILE "$line";
		    }
		}
		
		close (FILE);
	    }
	}
	
    }
    
}

# A sub-routine to delete
sub delUserVirtualSite {

    my $uid = shift || "";
    my $gid = shift || "";

    my %system = config::system();

    my %userInfo = datastore::userInfo($uid);
    my %groupInfo = datastore::groupInfo($gid);
    my $group = $groupInfo{'shortname'};

  general::debug("Group number = $gid");

    my $command = "/bin/cat $system{'datastore'}/groups.xml | /bin/grep \\\<UserID\\\>$userInfo{'uid'}\\\<\\\/UserID\\\> | /usr/bin/wc -l";
    my $result = `$command`;
    $command = "/bin/cat $system{'datastore'}/virtualsites.xml | /bin/grep \\\<ServerAdmin\\\>$userInfo{'uid'}\\\<\\\/ServerAdmin\\\>";
    my $serveradmin = `$command`;
    $result =~ s/\W//g;
    
  general::debug("User $userInfo{'uid'} used in \"$result\" places.  scalar = ". scalar($result) . "");

    # First check to make sure that they are not involved elsewhere
    if (scalar($result) <= 2 && $serveradmin eq "") {

      general::debug("Calling cust user delete");

      users::delUser("$uid","$gid");
	
    }
    else {

      general::debug("Deleting user from virtual site group $group only");

	# Update the group.xml file
	eval {
	    
	    # Set error handling
	    local $SIG{ALRM} = sub { 
		close(LOCK_FILE);
		print "<H3><FONT color='red'>ERROR:  Add group timed out.</FONT></H3>";
		warn("Add group timed out");
		die("alarm\n");
	    };
	    
	    # Set timeout
	    alarm($system{'locktimeout'});

	    # Set an exclusive lock
	    open(LOCK_FILE, ">$system{'datastore'}/groups.lck");

	    my $parser = new XML::DOM::Parser;
	    my $doc = $parser->parsefile("$system{'datastore'}/groups.xml");

	    # Search for this group
	    my $foundNode = "";
	    foreach my $groupNode ($doc->getElementsByTagName("Group")) {

		my $thisName = eval { $groupNode->getElementsByTagName("ShortName")->item(0)->getFirstChild->getData(); } || "";

		if ($thisName eq $group) {
		    $foundNode = $groupNode;
		    last;
		}
	    }

	    if ($foundNode ne "") {

		foreach my $memberNode ($foundNode->getElementsByTagName("UserID")) {

		    my $userid = eval { $memberNode->getFirstChild->getData(); } || "";

		    if ($userid eq $userInfo{'uid'}) {

			my $parentNode = $memberNode->getParentNode;
		    
			eval { $parentNode->removeChild($memberNode); };

		    }
		}

		$doc->printToFile("$system{'datastore'}/groups.xml");

	    }

	    close(LOCK_FILE);

	};
    }

    return();

}

# A sub-routine to turn on/off any module
sub setApacheModule {

    my $module = shift || return;
    my $status = lc shift || "on";

    my %system = config::system();

    # Parse out the Apache configuration
    my $parser = new XML::DOM::Parser;
    my $doc = $parser->parsefile("$system{'datastore'}/apache.xml");

    # Activate or de-activate the module
    if ($status eq "off" || $status eq "no") {
      datastore::ApacheRemoveModule ($doc, $module);
    }
    else {
      datastore::ApacheAddModule ($doc, $module);
    }

    # Store the results
  datastore::XMLtoFile ($doc, "$system{'datastore'}/apache.xml");

    return(1);

}


1;
