package network;
#
#  Package:      network.pm
#
#  Copyright:     COPYRIGHT Compaq Computer Corporation 2001.  All rights reserved.
#
#  Description:  The OS specific routines for changing the system.
#
#    These routines handle NETWORK specific changes including:
#
#     - IP Configuration
#     - SNMP Configuration
#     - DNS Setup
#     - etc.
#
#  History:
#
use strict;

use CGI;
use XML::DOM;
use XML::QL;
use Quota;
use Time::localtime;

attemptUse("apache.pm");
attemptUse("firewall.pm");

use config;
use general;
use datastore;
use message;
use error;

# A sub-routine to change a network interface
sub updateNetwork {
 
    my %system = config::system();
 
    my $configFile = "$system{'sysconfig'}/network";
 
    # First grab all IP addresses
 
    my $parser = new XML::DOM::Parser;
    my $doc = $parser->parsefile("$system{'datastore'}/network.xml");
 
    my $interfaces = $doc->getElementsByTagName("Network",0)->item(0)->getElementsByTagName("Interfaces",0)->item(0)->getElementsByTagName("Interface",0);
    my $numOfInterfaces = $interfaces->getLength;
    my $array = "";
 
    my $fqdn = eval { $doc->getElementsByTagName("Network",0)->item(0)->getElementsByTagName("HostName",0)->item(0)->getFirstChild->getData(); } || "";
    $fqdn =~ s/\n//g;

    for (my $i=0; $i<$numOfInterfaces;$i++) {
 
        my $interface = eval { $interfaces->item($i)->getElementsByTagName("IP")->item(0)->getFirstChild->getData(); } || "";
 
        $array .= "$interface<TD>";
    }
 
    # Split out the values
    my @ips = split(/<TD>/,$array);
 
    $ips[0] =~ s/ //g;
    $ips[1] =~ s/ //g;
 
    # Check to see if we've found the interface
    if ($fqdn eq "") {
        print "No hostname information found!";
        return(0);
    }
    else {
 
        # Add a notice that the file was modified by the Compaq Configurator
        my $date = `/bin/date`;
        my $message = "# File Modified by Compaq Configurator - $date";

	my @lines = ();
 
        # Read current configuration
        if (open(FILE, "<$configFile")) {
            @lines = <FILE>;
        }
 
        my $numOfLines = scalar(@lines);
        my $hasHostName = 0;
 
        for (my $i=0; $i<$numOfLines; $i++) {
            if ($lines[$i] =~ m/^HOSTNAME=/) {
                @lines[$i] = "HOSTNAME=$fqdn\n";
                $hasHostName = 1;
            }
        }

        # Add a hostname entry if one does not exist
        if (!$hasHostName) {
            push (@lines, "HOSTNAME=$fqdn\n");
        }
 
        # Write the file back out
        if (open(FILE,">$configFile")) {
            foreach my $line (@lines) {
                print FILE "$line";
            }
            close(FILE);
        }
 
        # Set hostname
        if (open(FILE,"</etc/hosts")) {
 
            @lines = <FILE>;
 
            close(FILE);
 
            my ($hostname) = split('\.', $fqdn);
	    my $hasLocalAddr = 0;
            my $hasIP0 = 0;
            my $hasIP1 = 0;
 
            if (open(FILE,">/etc/hosts")) {

	      general::debug("Opened hosts file for WRITING!!!!");
 
                # Add a notice that the file was modified by the Compaq Configurator
                my $message = "# File Modified by Compaq Configurator";
                my $date = `/bin/date`;
 
                print FILE "$message - $date
127.0.0.1\tlocalhost.localdomain\tlocalhost
127.0.0.2\t$fqdn\t$hostname
";
 
                foreach my $line (@lines) {
 
                    if ($line =~ m/File Modified by Compaq Configurator/) {
                        next;
                    }
                    elsif ($line =~ m/^127\.0\.0\.1/) {
                        next;
                    }
                    elsif ($line =~ m/^127\.0\.0\.2/) {
                        next;
                    }
                    elsif ($line eq "\n") {
                        next;
                    }
                    elsif ($line =~ m/^$ips[0]/ && $ips[0] ne "") {
                        $line = "$ips[0]\t$fqdn\t$hostname\n";
                        $hasIP0 = 1;
                    }
                    elsif ($line =~ m/^$ips[1]/ && $ips[1] ne "") {
                        $line = "$ips[1]\t$fqdn\t$hostname\n";
                        $hasIP1 = 1;
                    }
		    elsif ($line =~ m/$hostname/) {
			next;
		    }
 
                    print FILE "$line";
                }

		general::debug("IPS $ips[0] $ips[1]");

		if ($hasIP0 != 1 && $ips[0] ne "") {
                    print FILE "$ips[0]\t$fqdn\t$hostname\n";
                }
                if ($hasIP1 != 1 && $ips[1] ne "") {
                    print FILE "$ips[1]\t$fqdn\t$hostname\n";
                }
 
                close(FILE);
            }
 
            my $command = "/bin/hostname $hostname";
            my $result = `$command`;
 
            # Cycle the CIM agents and SNMP to have hostname take effect
            $result = `$system{'root'}/system/bin/snmpd stop`;
            $result = `$system{'root'}/system/bin/snmpd start`;
            $result = `$system{'root'}/system/bin/cmafdtn stop`;
            $result = `$system{'root'}/system/bin/cmafdtn start`;

	    # Remount any NFS partitions (IP change causes disconnect)
	    $result = `$system{'root'}/system/bin/umount -a -t nfs`;
	    $result = `$system{'root'}/system/bin/mount -a -t nfs`;
 
        }
 
    }
 
    return(1);
 
}
 
# A sub-routine to change a network interface
sub updateInterface {
 
    $| = 1;
 
    my $interface = shift || return(0);
    my $if = $interface;
 
    my %system = config::system();
 
    my $gateway = "";

    # Grab the old Interface IP addresses
    my $oldIP = "";
    my $newIP = "";
 
    # Choose corresponding config file for interface
    my $configFile = "$system{'netscripts'}/ifcfg-$if";
 
    # Parse the xml
    my $parser = new XML::DOM::Parser;
    my $doc = $parser->parsefile("$system{'datastore'}/network.xml");
 
    my $interfaces = $doc->getElementsByTagName("Network",0)->item(0)->getElementsByTagName("Interfaces",0)->item(0)->getElementsByTagName("Interface",0);
    my $numOfInterfaces = $interfaces->getLength;
 
    my $result = "";
 
    for (my $x=0; $x<$numOfInterfaces;$x++) {
 
        my $intif = eval {$interfaces->item($x)->getAttribute("if");} || "";
 
        if ($intif eq $if) {
	    my $enabled1 = eval {$interfaces->item($x)->getElementsByTagName("Enabled")->item(0)->getFirstChild->getData();} || "";
            my $mode1 = eval {$interfaces->item($x)->getElementsByTagName("Mode")->item(0)->getFirstChild->getData();} || "";
            my $ip1 = eval {$interfaces->item($x)->getElementsByTagName("IP")->item(0)->getFirstChild->getData();} || "";
            my $netmask1 = eval {$interfaces->item($x)->getElementsByTagName("NetMask")->item(0)->getFirstChild->getData();} || "";
            my $network1 = eval {$interfaces->item($x)->getElementsByTagName("Network")->item(0)->getFirstChild->getData();} || "";
            my $bcast1 = eval {$interfaces->item($x)->getElementsByTagName("Enabled")->item(0)->getFirstChild->getData();} || "";
 
            $result = "$enabled1<TD>$ip1<TD>$mode1<TD>$netmask1<TD>$network1<TD>$bcast1";
 
            last;
        }
    }

    # Grab the default gateway
    $gateway = eval {$doc->getElementsByTagName("Network",0)->item(0)->getElementsByTagName("DefaultGateway",0)->item(0)->getFirstChild->getData();} || ""; 

    # Check to see if we've found the interface
    if ($result eq "") {
	print "No interface configuration information found for $if!";
	return(0);
    }
    else {
 
        my @values = split(/<TD>/, $result);
 
        my $enabled = $values[0] || "No";
        my $ip = $values[1] || "";
        my $mode = $values[2] || "";
        my $netmask = $values[3] || "";
        my $network = $values[4] || "";
        my $bcast = $values[5] || "";
 
	# Grab the current configuration for this interface
	my %interfaceConfig = getCurrentIFConfig($interface);

        $oldIP = $interfaceConfig{'ip'};
        $newIP = $ip;
 
      general::debug("IP address used to be $oldIP and now it will be $newIP");
      general::debug( "Extracted GW of $gateway");
 
        # Add a notice that the file was modified by the Compaq Configurator
        my $message = "# File Modified by Compaq Configurator";
        my $date = `/bin/date`;
 
        my @lines = ();
        $lines[0] = "$message - $date";
        $lines[1] = "DEVICE=$if";
 
        if ($enabled eq "Yes") {
            push(@lines, "ONBOOT=yes");
        }
        else {
            push(@lines, "ONBOOT=no");
        }
 
        if ($mode eq "DHCP") {
            push(@lines, "BOOTPROTO=dhcp");
        }
        else {
            push(@lines, "IPADDR=$ip");
            push(@lines, "NETMASK=$netmask");
            push(@lines, "NETWORK=$network");
            push(@lines, "BROADCAST=$bcast");
            push(@lines, "GATEWAY=$gateway");
	  general::debug( "Pushed gateway onto new lines $gateway (Enabled=$enabled)");
        }
 
#       push(@lines, "USERCTL=yes");
 
        # Write the file back out
        if (open(FILE,">$configFile")) {
 
            foreach my $line (@lines) {

		print FILE "$line\n";
#               general::debug( "  $line");
            }
 
            close(FILE);
 
        }
 
        # Stop that interface
        my $command = "/sbin/ifconfig $if down";
        `$command`;
 
        # Restart only if enabled
        if ($enabled eq "Yes") {
 
            my $command = "";
 
            if ($mode eq "DHCP" || $mode eq "BOOTP") {
                my $ethpid = `/bin/cat /var/run/dhcpd-$if.pid`;
                $command = "/bin/kill -9 $ethpid";
		`$command`;
                `/bin/rm /var/run/dhcpd-$if.pid`;
                $command = "/sbin/dhcpcd -B -r -t 30 -i TaskSmartW2200 $if";
                `$command`;
 
                # Grab cached information in case dhcp server didn't respond
                my $ip = `/bin/cat /etc/dhcpc/dhcpcd-$if.info | /bin/grep IPADDR`;
                my $netmask = `/bin/cat /etc/dhcpc/dhcpcd-$if.info | /bin/grep NETMASK`;
 
                $ip =~ s/\n//g;
                $ip =~ s/IPADDR=//g;
                $netmask =~ s/\n//g;
                $netmask =~ s/NETMASK=//g;
 
                `/sbin/ifconfig $if up netmask $netmask $ip`;
            }
            else {
                # We execute this twice because of a little bug in ifconfig
                $command = "/sbin/ifconfig $if up netmask $netmask $ip";
                `$command`;
		`$command`;
            }
 
            # Reset the default gateway
            if ($gateway ne "") {
 
	      general::debug( "Manually adding default gw of $gateway");
                $command = "/sbin/route -n del default";
                $result = `$command`;
                $command = "/sbin/route -n add default gw $gateway";
                $result = `$command`;
            }
 
            # Do post configuration updates to the XML
            my $ifconfig = `/sbin/ifconfig $if`;
            my $route = `/sbin/route -n | /bin/grep ^0.0.0.0`;
            my @values = split(/\s*/, $route);
            $gateway = $values[1];
 
            $ifconfig =~ /inet addr:([\d\.]*)/;
	    my $ip = $1;
            $ifconfig =~ /Mask:([\d\.]*)/;
            my $netmask = $1;
            $ifconfig =~ /Bcast:([\d\.]*)/;
            my $bcast = $1;
            $ifconfig =~ m/HWaddr ([\dABCDEF]{2}:[\dABCDEF]{2}:[\dABCDEF]{2}:[\dABCDEF]{2}:[\dABCDEF]{2}:[\dABCDEF]{2})/;
            my $macaddr = $1;
 
            my $interfaceNode = "";
 
            for (my $i=0; $i<$numOfInterfaces; $i++) {
 
                my $thisIF = eval { $interfaces->item($i)->getAttribute("if"); } || "";
 
                # Did we find the right node?
                if ($thisIF eq $if) {
                    $interfaceNode = $interfaces->item($i);
                }
	    }
 
            if ($interfaceNode eq "") {
                print "<H1><FONT color=red>Unknown interface: $interface</FONT></H1>";
            }
            else {
 
              datastore::textNodeUpdate($interfaceNode->getElementsByTagName("MACAddress")->item(0), "$macaddr");
              datastore::textNodeUpdate($interfaceNode->getElementsByTagName("IP")->item(0), "$ip");
              datastore::textNodeUpdate($interfaceNode->getElementsByTagName("NetMask")->item(0), "$netmask");
              datastore::textNodeUpdate($interfaceNode->getElementsByTagName("Broadcast")->item(0), "$bcast");
              datastore::textNodeUpdate($interfaceNode->getElementsByTagName("DefaultGateway")->item(0), "$gateway");
 
                # Write out the file
                $doc->printToFile("$system{'datastore'}/network.xml");

		$doc->dispose();
            }
        }
        # Really disable the interface
        else {
            my $command = "/sbin/ifconfig $if 0.0.0.0";
            `$command`;
        }
    }

    # Update the hosts file
    updateNetwork();
 
    # Update the virutal sites to reflect the new IP address
    if ($INC{'firewall.pm'}) {
      general::debug("Firewall was loaded, i'm updating the services");
      firewall::updateServices();
    }

    # Update apache if loaded
    if ($INC{'apache.pm'}) {
      general::debug("Apache was loaded, i'm reseting the IP's");
      apache::updateVirtualSiteIPs($oldIP, $newIP);
      apache::restartApache();
    }
 
    return(1);
 
}

# A sub-routine to configure name resolution
sub updateNameResolution {
 
    my %system = config::system();
 
    # Step 1: Re-configure the name servers
 
    #  Parse the network.xml file
 
    my $parser = new XML::DOM::Parser;
    my $doc = $parser->parsefile("$system{'datastore'}/network.xml");
 
    my $dfltsrchdomain = eval { $doc->getElementsByTagName("Network",0)->item(0)->getElementsByTagName("DefaultSearchDomain",0)->item(0)->getFirstChild->getData();} || "";
 
    my $nmeservers = eval { $doc->getElementsByTagName("Network",0)->item(0)->getElementsByTagName("NameServers",0)->item(0)->getElementsByTagName("NameServer",0); } || "";
    my $numOfServers = $nmeservers->getLength;
 
    my $serverarray = "";
 
    for (my $j=0; $j<$numOfServers; $j++) {
 
        my $serveNode = $nmeservers->item($j);
	my $value = eval {$serveNode->getFirstChild->getData();} || "";
 
        $serverarray .= "$value<TD>";
    }
 
    my @nameservers = split("<TD>", $serverarray);
 
    my $configFile = "/etc/resolv.conf";
 
    # Read current entries
    my @lines = ();
    if (open(FILE,"<$configFile")) {
        @lines = <FILE>;
        close(FILE);
    }
 
    my @newlines = ();
 
    # Write "resolv.conf" back out
    if (open(FILE, ">$configFile")) {
 
        my $numOfLines = scalar(@lines);
 
        for (my $i=0; $i<$numOfLines; $i++) {
 
            # Keep commented lines
            if ($lines[$i] =~ m/^#/) {
                push (@newlines, $lines[$i]);
                next;
            }
            # Remove the default search domain
            elsif ($lines[$i] =~ m/^search/) {
                next;
            }
            # Remove existing name servers
            elsif ($lines[$i] =~ m/^nameserver/) {
                next;
            }
 
        }

	# Add the default search domain
        push(@newlines, "search $dfltsrchdomain\n");
 
        foreach my $nameserver (@nameservers) {
            push(@newlines, "nameserver $nameserver\n");
        }
 
        # Write out the file
        foreach my $line (@newlines) {
            print FILE "$line";
        }
 
        close(FILE);
 
    }

    # Step 2: Update static "hosts" entry
    my $hosts = $doc->getElementsByTagName("Host");
    my $numOfHosts = $hosts->getLength;
 
    my $hostarray = "";
 
    for (my $h=0; $h<$numOfHosts; $h++) {
 
        my $hostNode = $hosts->item($h);
        my $name = eval {$hostNode->getAttribute("name");} || "";
        my $ip = eval {$hostNode->getFirstChild->getData();} || "";
 
        $hostarray .= "$name<TD>$ip<TR>";
    }
 
    my @rows = split('<TR>',$hostarray);
 
    $configFile = "/etc/hosts";
 
    # Read current entries
    @lines = ();
    if (open(FILE,"<$configFile")) {
        @lines = <FILE>;
        close(FILE);
    }
 
    my $date = `/bin/date`;
    if ($lines[0] =~ m/# File Modified by Compaq Configurator/) {
        $lines[0] = "# File Modified by Compaq Configurator - $date\n";
    }
    else {
        push(@lines, "# File Modified by Compaq Configurator - $date\n");
    }
 
    @newlines = ();
 
    # Write "hosts" back out
    if (open(FILE, ">$configFile")) {
	
        my $numOfLines = scalar(@lines);
	
        for (my $i=0; $i<$numOfLines; $i++) {
	    
            # Keep commented lines
            if ($lines[$i] =~ m/^#/) {
                push (@newlines, $lines[$i]);
                next;
            }
            # Preserve local hosts
            elsif ($lines[$i] =~ m/^127/) {
                push (@newlines, $lines[$i]);
                next;
            }
            # Remove all other lines
            else {
                next;
            }
	    
        }
	
	my $fqdn = eval {$doc->getElementsByTagName("HostName")->item(0)->getFirstChild->getData();} || "";
	$fqdn =~ s/\n//g;
  
        my $interfaces = $doc->getElementsByTagName("Interfaces")->item(0)->getElementsByTagName("Interface",0);
        my $numOfInterfaces = $interfaces->getLength;
 
        my $iparray = "";
 
        for (my $int=0; $int < $numOfInterfaces; $int++) {
 
            my $intNode = $interfaces->item($int);
            my $ipelement = eval {$intNode->getElementsByTagName("IP")->item(0)->getFirstChild->getData();} || "";
 
            $iparray .= "$ipelement<TD>";
        }
 
        # Split out the values
        my @ips = split(/<TD>/,$iparray);
	my @pieces = split('\.', $fqdn);
	pop(@pieces);
	pop(@pieces);
	my $hostname = join('\.', @pieces);
 
        foreach my $ip (@ips) {
            $ip =~ s/ //g;
            push(@newlines, "$ip\t\t$fqdn\t\t$hostname\n");
        }
 
 
        foreach my $row (@rows) {
            my ($name, $ip) = split('<TD>',$row);
 
            # Add the new static hosts
            push(@newlines, "$ip\t\t$name\n");
        }
 
        # Write out the file
        foreach my $line (@newlines) {
            print FILE "$line";
        }
 
        close(FILE);
 
    }
 
 
}

# A sub-routine to update the routing table
sub updateRouting {
 
    my %system = config::system();
 
    my $configFile = "/etc/sysconfig/static-routes";
    
    if (open(FILE, ">$configFile")) {
	
        my $parser = new XML::DOM::Parser;
        my $doc = $parser->parsefile("$system{'datastore'}/network.xml");
	
        my $dfgw = eval { $doc->getElementsByTagName("DefaultGateway")->item(0)->getFirstChild->getData(); } || "";
	
        my $result = "";
	

        # Step 2: Grab the routes
        foreach my $routeNode ($doc->getElementsByTagName("Network",0)->item(0)->getElementsByTagName("Route",0)) {
	    
            my $type = eval { $routeNode->getElementsByTagName("RouteType",0)->item(0)->getFirstChild->getData(); } || "";
	    my $dest = eval { $routeNode->getElementsByTagName("Destination",0)->item(0)->getFirstChild->getData(); } || "";

	    my $gw = eval { $routeNode->getElementsByTagName("Gateway",0)->item(0)->getFirstChild->getData(); } || "";
	    my $netmask = eval { $routeNode->getElementsByTagName("NetMask",0)->item(0)->getFirstChild->getData(); } || "";
	    my $if = eval { $routeNode->getElementsByTagName("Interface",0)->item(0)->getFirstChild->getData(); } || "";
	    $if = lc $if;
	    
	    $result .= "$type<TD>$dest<TD>$gw<TD>$netmask<TD>$if<TD><TR>";
	}

	$doc->dispose();
	
	my @rows = split('<TR>', $result);
	
	my $date = `/bin/date`;
	my $message = "# File Modified by Compaq Configurator - $date";
	
	print FILE "$message\n";

	my $gateway = "";
	my $command = "";

    	# Cycle the network cards
    	my $command = "/sbin/ifconfig eth0 down";
    	`$command`;	
    	$command = "/sbin/ifconfig eth1 down";
    	`$command`;	
	
    	$command = "/sbin/ifconfig eth0 up";
    	`$command`;	
    	$command = "/sbin/ifconfig eth1 up";
    	`$command`;

#  	general::debug( "Cycled network cards!");

	# Delete each route
	my $command = "/sbin/route -n";
	my $result = `$command`;

	my @currroutes = split('\n', $result);
	shift(@currroutes);

	foreach my $delroute (@currroutes) {

	    my @values = split(' ', $delroute);
	    $command = "/sbin/route -n del $values[0] gw $values[1]";
#	    general::debug( " DELETE ROUTE = $command");
	    $result = `$command`;
	}

	# Process each route
	foreach my $row (@rows) {

	    my ($type, $dest, $gw, $netmask, $if) = split('<TD>', $row);

	    if ($gateway eq "" && $gw ne "") {
		$gateway = $gw;
	    }

	    print FILE "$if $type $dest netmask $netmask gw $gw\n";

	    if ($type eq "net") {
		$command = "/sbin/route -n add -net $dest netmask $netmask gw $gw dev $if";
		$result = `$command`;
	    }
	    else {
		$command = "/sbin/route -n add -host $dest gw $gw dev $if";
		$result = `$command`;
	    }

#	    general::debug( "ROUTING COMMAND = $command");


	}

	# Add the default gateway back into the routing table
	$dfgw =~ s/ //g;
	$command = "/sbin/route -n add default gw $dfgw";
	$result = `$command`;

	general::debug( "Added default gateway; command = $command  ($dfgw)!");

	close (FILE);
    
    }
}


# A sub-routine to update the SNMP settings for the system
sub updateSNMPConfig {

    my %system = config::system();

    # Step 1:  Parse information about SNMP settings
    my $parser = new XML::DOM::Parser;
    my $doc = $parser->parsefile("$system{'datastore'}/network.xml");

    # Grab the values for the log files
    my $rocomm = eval { $doc->getElementsByTagName("Network",0)->item(0)->getElementsByTagName("SNMP",0)->item(0)->getElementsByTagName("ROCommunity")->item(0)->getFirstChild->getData(); } || "";
    my $rwcomm = eval { $doc->getElementsByTagName("Network",0)->item(0)->getElementsByTagName("SNMP",0)->item(0)->getElementsByTagName("RWCommunity")->item(0)->getFirstChild->getData(); } || "";
    my $trapcomm = eval { $doc->getElementsByTagName("Network",0)->item(0)->getElementsByTagName("SNMP",0)->item(0)->getElementsByTagName("TrapCommunityName")->item(0)->getFirstChild->getData(); } || "";
    my $trapaddr = eval { $doc->getElementsByTagName("Network",0)->item(0)->getElementsByTagName("SNMP",0)->item(0)->getElementsByTagName("TrapServerAddress")->item(0)->getFirstChild->getData(); } || "";

    # Step 2: Update the snmp.conf file
    if (open (FILE, "</usr/share/snmp/snmpd.conf")) {

	general::debug( "SNMP Update - Read in the snmpd.conf file");

	my @lines = <FILE>;

	close (FILE);

	if (open (FILE, ">/usr/share/snmp/snmpd.conf")) {

	    my $addrocomm = 0;
	    my $addrwcomm = 0;
	    my $addtrapcomm = 0;
	    my $addtrapaddr = 0;

	    foreach my $line (@lines) {

		general::debug( "$line");

		# Find the RO community string, but avoid the Compaq setup
		if ($line =~ m/rwcommunity.*localcomm/) {
		    print FILE "$line";
		}
		elsif ($line =~ m/^rocommunity/) {
		    if ($rocomm ne "") {
			$line = "rocommunity $rocomm\n";
			print FILE "$line";
			$addrocomm = 1;
		    }
		}
		elsif ($line =~ m/^rwcommunity/) {
		    if ($rwcomm ne "") {
			$line = "rwcommunity $rwcomm\n";
			print FILE "$line";
			$addrwcomm = 1;
		    }
		}
		# Update the trap server
		elsif (($line =~ m/^trapsink/) && !($line =~ m/127.0.0.1/)) {
		    if ($trapaddr ne "") {
			$line = "trapsink $trapaddr\n";
			print FILE "$line";
			$addtrapaddr = 1;
		    }
		}
		elsif ($line =~ m/^trapcommunity/) {
		    if ($trapcomm ne "") {
			$line = "trapcommunity $trapcomm\n";
			print FILE "$line";
			$addtrapcomm = 1;
		    }
		}
		else {
		    print FILE "$line";
		}
	    }

	    # Add community lines if they don't exist
	    if ($addrocomm != 1 && $rocomm ne "") {
		print FILE "rocommunity $rocomm\n";
	    }
	    if ($addrwcomm != 1 && $rwcomm ne "") {
		print FILE "rwcommunity $rwcomm\n";
	    }
	    if ($addtrapcomm != 1 && $trapcomm ne "") {
		print FILE "trapcommunity $trapcomm\n";
	    }
	    if ($addtrapaddr != 1 && $trapaddr ne "") {
		print FILE "trapsink $trapaddr\n";
	    }

	    close (FILE);

	    general::debug( "Wrote snmpd.conf file out.  [$rocomm, $rwcomm, $trapcomm, $trapaddr] ($addrocomm, $addrwcomm, $addtrapcomm, $addtrapaddr)");
	}
    }

    # Cycle snmp daemon
    my $command = "/usr/bin/killall -HUP snmpd";
    my $result = `$command`;

}

# Get the date/time of the SSL files
#  Return key~csr~crt
sub getApacheSSLdates {

    my $result = `/bin/ls -l --full-time /etc/httpd/conf/ssl.key/server.key`;
    my @values = split(' ', $result);
    my $return = "$values[5] $values[6] $values[7] $values[8] $values[9] ";

    $result = `/bin/ls -l --full-time /etc/httpd/conf/ssl.csr/server.csr`;
    @values = split(' ', $result);
    $return .= "~$values[5] $values[6] $values[7] $values[8] $values[9] ";

    $result = `/bin/ls -l --full-time /etc/httpd/conf/ssl.crt/server.crt`;
    @values = split(' ', $result);
    $return .= "~$values[5] $values[6] $values[7] $values[8] $values[9] ";

    return ($return);

}

# A sub-routine to add an NFS mount point to the system
sub addNFSMount {

    my $connectString = shift || return;
    my $mountPoint = shift || return;

    general::debug( "NFS add - $connectString\t$mountPoint");
    
    my %system = config::system();

    # Safe-gaurd against shell manipulation
    $connectString =~ s/;//;
    $mountPoint =~ s/;//;
    
    # Step 1 - Check to see if the directory exists and
    # make the directory if it does not exist.
    if (!(-d "$mountPoint")) {
	
	my $command = "/bin/mkdir $mountPoint";
	my $result = `$command`;
	
	if (!(-d "$mountPoint")) {
	    return("MOUNTPOINT_COULDNOTCREATE");
	}

    }
    
    # Step 2 - Add the mount to /etc/fstab
    my @lines = ();
    if (open(FILE,"</etc/fstab")) {
	
	@lines = <FILE>;
	
	close(FILE);

	# Add the new NFS mount
	if (open(FILE, ">/etc/fstab")) {
	    
	    foreach my $line (@lines) {

		# Don't duplicate the NFS mount
		if ($line =~ m/$connectString \t $mountPoint/) {
		    next;
		}
		else {
		    print FILE $line;
		}
	    }
	}

	print FILE "$connectString \t $mountPoint \t nfs \t user,rsize=8192,wsize=8192,timeo=14,intr\n";	
    }

    # Step 3 - Attempt to manually mount the NFS mount point
    my $command = "/bin/mount $mountPoint";
    my $result = `$command`;

    general::debug( "$result");

    # Step 4 - Fix-up for tighter security
    my @lines = ();
    if (open(FILE,"</etc/fstab")) {
	
	@lines = <FILE>;
	
	close(FILE);

	# Search for the previosuly added line and fix it
	if (open(FILE, ">/etc/fstab")) {
	    
	    foreach my $line (@lines) {

		if ($line eq "$connectString \t $mountPoint \t nfs \t user,rsize=8192,wsize=8192,timeo=14,intr\n") {
		    print FILE "$connectString \t $mountPoint \t nfs \t rsize=8192,wsize=8192,timeo=14,intr\n";
		}
		else {
		    print FILE $line;
		}
	    }
	}
    }
    
    return("");
    
}

# A sub-routine to delete an NFS mount point
sub delNFSMount {

    my %system = config::system();

    my $connectString = shift || return;
    my $mountPoint = shift || return;

    # Safe-gaurd against shell manipulation
    $connectString =~ s/;//;
    $mountPoint =~ s/;//;

    # Step 1 - Attempt to manually un-mount the NFS mount point
    my $command = "$system{'root'}/system/bin/umount $mountPoint";
    my $result = `$command`;

    # Step 2 - Fix-up the fstab to allow user dis-mount
    my @lines = ();
    if (open(FILE,"</etc/fstab")) {

	@lines = <FILE>;

	close(FILE);
    }

    if (open(FILE, ">/etc/fstab")) {

	foreach my $line (@lines) {

	    # Skip the line we are tring to remove
	    if ($line =~ m/^$connectString(\W+)$mountPoint(\W+)/) {
		next;
	    }
	    else {
		print FILE $line;
	    }

	}
    }

    $command = "/bin/rmdir $mountPoint";
    $result = `$command`;

    return("");

}

# A sub-routine to update the NFS mounts as described in the datastore
sub updateNFSMounts {

    my %system = config::system();

    my $parser = new XML::DOM::Parser;
    my $doc = $parser->parsefile("$system{'datastore'}/network.xml");

    foreach my $mount ($doc->getElementsByTagName("Network", 0)->item(0)->getElementsByTagName("Mount", 0)) {

	my $remote = eval { $mount->getElementsByTagName("Remote",0)->item(0)->getFirstChild->getData(); } || "";
	my $local = eval { $mount->getElementsByTagName("Local",0)->item(0)->getFirstChild->getData(); } || "";

	addNFSMount($remote, $local);

    }

    $doc->dispose();

    return(1);
}

# Update the SSH settings
sub updateSSH {

    my %system = config::system();

    my $regenkey = shift || "";

    if ($regenkey eq "Yes") {
	my $command = "/bin/rm /etc/ssh/ssh_host_key";
	my $result = `$command`;
	$command = "/bin/rm /etc/ssh/ssh_host_key.pub";
	$result = `$command`;
	$command = "/usr/bin/ssh-keygen -q -f /etc/ssh/ssh_host_key";
	$result = `$command`;
#	general::debug( "SSH Key Rebuild Command = $command");
#	general::debug( "Result = $result");
    }

    # Step 1:  Grab SSH settings

    my $parser1 = new XML::DOM::Parser;
    my $doc1 = $parser1->parsefile("$system{'datastore'}/network.xml");

    my $setup = $doc1->getElementsByTagName("SSHSetup")->item(0);
    my $rootlogin = eval {$setup->getElementsByTagName("RootLogin")->item(0)->getFirstChild->getData();} || "";
    my $interval = eval {$setup->getElementsByTagName("KeyRegenInterval")->item(0)->getFirstChild->getData();} || "";


    # Step 2:  REMOVED
    # Step 3:  Update sshd_config file
    if (open(FILE, "</etc/ssh/sshd_config")) {

	my @lines = <FILE>;

	close(FILE);

	if (open (FILE, ">/etc/ssh/sshd_config")) {

	    foreach my $line (@lines) {

		# Change permit root login
		if ($line =~ m/^PermitRootLogin/) {
		    if ($rootlogin eq "Yes") {
			$line = "PermitRootLogin yes\n";
		    }
		    else {
			$line = "PermitRootLogin no\n";
		    }
		}

		# Change update interval
		if ($line =~ m/^KeyRegenerationInterval/ && $interval ne "") {
		    $line = "KeyRegenerationInterval $interval\n";
		}

		print FILE "$line";
	    }

	    close (FILE);
	}
    }

    # Step 4;  Cycle sshd
    my $command = "";
    if (open(FILE, "</var/run/sshd.pid")) {

	my $PID = <FILE>;
	close(FILE);

	$command = "/bin/kill -s HUP $PID";

	general::debug( "Simple HUP restart");

    }

    # Check to make sure we are running
    my $processes = `/bin/ps auwx | /bin/grep sshd`;

    if (!($processes =~ m/\/usr\/sbin\/sshd/)) {

	general::debug( "Restarting SSHD the hard way!");

	# Stop the server
	$command = "/usr/bin/killall /usr/sbin/sshd";
	my $result = `$command`;

	# Remove any lock files
	$command = "/bin/rm -f /var/lock/subsys/sshd /var/run/sshd.pid";
	$result = `$command`;

	$command = "/usr/sbin/sshd";

    }

    # Cycle network services
    if ($INC{'firewall.pm'}) {
      firewall::updateServices();
    }

    general::debug( "Starting SSHD!! command = $command");

    general::debug( `$command`);

}

# A sub-routine to do a non-fatal include of packages
sub attemptUse {

    my ($filename) = @_;

    if ($INC{$filename}) {
      general::debug("Module already loaded $filename");
	return(1);
    }

    my ($realfilename, $result);

  ITER: {
      
      foreach my $prefix (@INC) {

	  $realfilename = "$prefix/$filename";

	  if (-f $realfilename) {
	
	    general::debug("Loading \"$realfilename\"");	
	      
	      $INC{$filename} = $realfilename;
	      $result = do $realfilename;

	      if (!$result) {
		  delete $INC{$filename}
	      }
	      last ITER;
	  }
      }

    general::debug("Not Loading \"$filename\".");
      
  }    
    
#    die $@ if $@;
#    die "$filename did not return true value" unless $result;

    return $result;
}

# A sub-routine to return the current IP address
sub getCurrentIFConfig {

    my $interface = shift || "eth0";

    my %ifConfig = "";

    my $result = `/sbin/ifconfig $interface`;

    my @lines = split(/\n/, $result);

    # Grab the IP address
    $lines[1] =~ m/addr:(\d*\.\d*\.\d*\.\d*)/;
    $ifConfig{'ip'} = $1;

    # Grab the netmask
    $lines[1] =~ m/Mask:(\d*\.\d*\.\d*\.\d*)/;
    $ifConfig{'netmask'} = $1;

    # Grab the broadcast
    $lines[1] =~ m/Bcast:(\d*\.\d*\.\d*\.\d*)/;
    $ifConfig{'broadcast'} = $1;

    # Grab MAC Addr
    $lines[0] =~ m/HWaddr (\d*\.\d*\.\d*\.\d*)/;
    $ifConfig{'macaddr'} = $1;

    return(%ifConfig);

    
}

1;
