package valid;
#
#  Package:      valid.pm
#
#  Description:  A collection of sub-routines for validating data.
#
#    This includes:
#     - Valid TCP/IP (IP Addr., netmask, gw's, etc.)
#     - Valid host/domain name
#     - Valid login name/password
#     - etc.
#   
#
#  History:	 Modularized July 18th, 2001
#
 
use strict;
 
use CGI;
use XML::DOM;
use XML::QL;
use Net::IPv4Addr qw( :all);
 
use config;

# A sub-routine to verify that a character is a numerical value
sub validInteger {
 
    my $value = shift || return;
 
    if ($value =~ m/[^\d\-]/) {
        return("INVALID_INTEGER_HASNONDIGITS");
    }
 
    return("");
}

# A sub-routine to verify that a quota limit is valid
sub validDiskQuota {
 
    my $quota = shift || return;

    if ($quota =~ m/^\-\d*/) {
        return("INVALID_QUOTA_LESSTHANZERO");
    } 
    elsif ($quota =~ m/\D/) {
        return("INVALID_QUOTA_HASNONDIGITS");
    }
 
    return("");
}

# A sub-routine to verify that a content user limit is valid
sub validContentAllotment {
 
    my $users = shift || return;
 
    if ($users =~ m/\D/) {
        return("INVALID_CONTENTALLOTMENT_HASNONDIGITS");
    }
    elsif ($users < 0) {
        return("INVALID_CONTENTALLOTMENT_LESSTHANZERO");
    }
 
    return("");
}

# A sub-routine to verify that a CPU limit is valid
sub validCPULimit {
 
    my $cpu = shift || return;
 
    if ($cpu =~ m/\D/) {
        return("INVALID_CPULIMIT_HASNONDIGITS");
    }
    elsif ($cpu < 0) {
        return("INVALID_CPULIMIT_LESSTHANZERO");
    }
 
    return("");
}

# A sub-routine to verify that a CPU limit is valid
sub validBandwidthLimit {
 
    my $bandwidth = shift || return;
 
    if ($bandwidth =~ m/\D/) {
        return("INVALID_BANDWIDTHLIMIT_HASNONDIGITS");
    }
    elsif ($bandwidth < 0) {
        return("INVALID_BANDWIDTHLIMIT_LESSTHANZERO");
    }
 
    return("");
}

# A sub-routine to verify that a memory limit is valid
sub validMemoryLimit {
 
    my $memory = shift || return;
 
    if ($memory =~ m/\D/) {
        return("INVALID_MEMORYLIMIT_HASNONDIGITS");
    }
    elsif ($memory < 0) {
        return("INVALID_MEMORYLIMIT_LESSTHANZERO");
    }
 
    return("");
}

# A sub-routine to verify that a process limit is valid
sub validProcessLimit {
 
    my $process = shift || return;
 
    if ($process =~ m/\D/) {
        return("INVALID_PROCESSLIMIT_HASNONDIGITS");
    }
    elsif ($process < 0) {
        return("INVALID_PROCESSLIMIT_LESSTHANZERO");
    }
 
    return("");
}

# A sub-routine to verify that an email address is valid
sub validEmail {
 
    my $email = shift || return;
    my @values = split('\@', $email);
    my $numOfSections = scalar(@values);
    my @dotsplit = split('\.', $values[1]);
    my $sectionTooLong = 0;
 
 
    if (length($dotsplit[0]) > 63) {
        $sectionTooLong = 1;
    }
 
    if (length($values[0]) > 63) {
        $sectionTooLong=1;
    }
 
    # Check for spaces
    if ($email =~ m/ /) {
        return("INVALID_EMAIL_HASSPACES");
    }
    # Check for invalid characters
    elsif ($email =~ m/[^\dA-Za-z\.\-\_\@]/) {
        return("INVALID_EMAIL_INVALIDCHARACTERS");
    }
    # Check for a digit at the beginning
    elsif ($email =~ m/^\d/) {
        return("INVALID_EMAIL_BEGINSWITHDIGIT");
    }
    # Check for proper length between the values
    elsif ($sectionTooLong ne "0") {
        return("INVALID_EMAIL_SECTION_TOOLONG");
    }
    # Check for proper use of @
    elsif ($numOfSections ne "2") {
     return("INVALID_EMAIL_BADFORMAT");
    }
    # Check for maximum length not to exceed 254
    elsif (length($email) > 254) {
        return("INVALID_EMAIL_TOOLONG");
    }
    # Check for proper suffix (.com, .net, etc)
    if (
        $dotsplit[(scalar(@dotsplit)-1)] ne "com" &&
        $dotsplit[(scalar(@dotsplit)-1)] ne "net" &&
        $dotsplit[(scalar(@dotsplit)-1)] ne "org" &&
        $dotsplit[(scalar(@dotsplit)-1)] ne "gov" &&
        $dotsplit[(scalar(@dotsplit)-1)] ne "edu" &&
	$dotsplit[(scalar(@dotsplit)-1)] ne "mil" &&
        $dotsplit[(scalar(@dotsplit)-1)] ne "int" &&
        $dotsplit[(scalar(@dotsplit)-1)] ne "biz" &&
        $dotsplit[(scalar(@dotsplit)-1)] ne "info" &&
        $dotsplit[(scalar(@dotsplit)-1)] ne "name" &&
        $dotsplit[(scalar(@dotsplit)-1)] ne "pro" &&
        $dotsplit[(scalar(@dotsplit)-1)] ne "museum" &&
        $dotsplit[(scalar(@dotsplit)-1)] ne "aero" &&
        $dotsplit[(scalar(@dotsplit)-1)] ne "coop" &&
        length($dotsplit[(scalar(@dotsplit)-1)]) != 2
        ) {
        return("INVALID_EMAIL_BAD_SUFFIX");
 
    }
 
    else {
        return("");
    }
  
}

# A sub-routine to verify that a group name is valid
sub validGroupName {
 
    my $groupname = shift || return;
 
    if ($groupname =~ m/ /) {
        return("INVALID_GROUPNAME_HASSPACES");
    }
    elsif ($groupname =~ m/^\d/) {
        return("INVALID_SHORTNAME_BEGINSWITHDIGIT");
    }
    elsif (scalar($groupname) >= 16) {
        return("INVALID_GROUPNAME_TOOLONG");
    }
 
    return("");
 
}

# A sub-routine to verify that an email alias name is valid
sub validAliasName {
 
    my $aliasname = shift || return;
 
    if ($aliasname =~ m/ /) {
        return("EMAILALIAS_HASSPACES");
    }
    if ($aliasname =~ m/@/) {
        return("EMAILALIAS_BADFORMAT");
    }
 
    # Check for invalid characters
    elsif ($aliasname =~ m/[^\dA-Za-z\.\-\_]/) {
        return("EMAILALIAS_INVALIDCHARACTERS");
    }
    # Check for a digit at the beginning
    elsif ($aliasname =~ m/^\d/) {
        return("EMAILALIAS_BEGINSWITHDIGIT");
    }
    # Check for maximum length not to exceed 254
    elsif (length($aliasname) > 254) {
        return("EMAILALIAS_TOOLONG");
    }
    return("");
    
}

# A sub-routine to verify that an IP address is valid
sub validHostName {
 
    my $hostname = lc shift || return;
 
    my @values = split('\.', $hostname);
    my $numOfSections = scalar(@values);
 
    my $sectionTooLong = 0;
    foreach my $value (@values) {
        if (length($value) > 63) {
            $sectionTooLong = 1;
        }
    }
 
    # Check for spaces
    if ($hostname =~ m/ /) {
        return("INVALID_HOSTNAME_HASSPACES");
    }
    # Check for invalid characters
    elsif ($hostname =~ m/[^\dA-Za-z\.\-]/) {
        return("INVALID_HOSTNAME_INVALIDCHARACTERS");
    }
    # Check for a digit at the beginning
    elsif ($hostname =~ m/^\d/) {
        return("INVALID_HOSTNAME_BEGINSWITHDIGIT");
    }
    # Check for proper length between the dots
    elsif ($sectionTooLong) {
	return("INVALID_HOSTNAME_SECTION_TOOLONG");
    }
    elsif ($numOfSections < 3) {
        return("INVALID_HOSTNAME_BADFORMAT");
    }
    # Check for maximum length not to exceed 254
    elsif (length($hostname) > 254) {
        return("INVALID_HOSTNAME_TOOLONG");
    }
    # Check for proper suffix (.com, .net, .org, .gov, .edu, .mil., .int, .??)
    if (
          $values[($numOfSections-1)] ne "com" &&
          $values[($numOfSections-1)] ne "net" &&
          $values[($numOfSections-1)] ne "org" &&
          $values[($numOfSections-1)] ne "gov" &&
          $values[($numOfSections-1)] ne "edu" &&
          $values[($numOfSections-1)] ne "mil" &&
          $values[($numOfSections-1)] ne "int" &&
          $values[($numOfSections-1)] ne "biz" &&
          $values[($numOfSections-1)] ne "info" &&
          $values[($numOfSections-1)] ne "name" &&
          $values[($numOfSections-1)] ne "pro" &&
          $values[($numOfSections-1)] ne "museum" &&
          $values[($numOfSections-1)] ne "aero" &&
          $values[($numOfSections-1)] ne "coop" &&
          length($values[($numOfSections-1)]) != 2
	) {
        return("INVALID_HOSTNAME_BAD_SUFFIX");
    }
    else {
        return("");
    }
 
}

# A sub-routine to verify that Site Name is valid for a VirtualSite
sub validVirtualSiteName {
 
    my $sitename = lc shift || return;
 
    my @values = split('\.', $sitename);
    my $numOfSections = scalar(@values);
 
    my $sectionTooLong = 0;
    foreach my $value (@values) {
        if (length($value) > 63) {
            $sectionTooLong = 1;
        }
    }
 
    # Check for spaces
    if ($sitename =~ m/ /) {
        return("INVALID_SITENAME_HASSPACES");
    }
    # Check for invalid characters
    elsif ($sitename =~ m/[^\dA-Za-z\.\-]/) {
        return("INVALID_SITENAME_INVALIDCHARACTERS");
    }
    elsif ($numOfSections < 2) {
        return("INVALID_SITENAME_BADFORMAT");
    }
    # Check for a digit at the beginning
    elsif ($sitename =~ m/^\d/) {
        return("INVALID_SITENAME_BEGINSWITHDIGIT");
    }
    # Check for proper length between the dots
    elsif ($sectionTooLong) {
        return("INVALID_SITENAME_SECTION_TOOLONG");
    }
    # Check for maximum length not to exceed 254
    elsif (length($sitename) > 254) {
        return("INVALID_SITENAME_TOOLONG");
    }
    # Check for proper suffix (.com, .net, .org, .gov, .edu, .mil., .int, .??)
    if (
          $values[($numOfSections-1)] ne "com" &&
          $values[($numOfSections-1)] ne "net" &&
          $values[($numOfSections-1)] ne "org" &&
          $values[($numOfSections-1)] ne "gov" &&
          $values[($numOfSections-1)] ne "edu" &&
          $values[($numOfSections-1)] ne "mil" &&
          $values[($numOfSections-1)] ne "int" &&
          $values[($numOfSections-1)] ne "biz" &&
          $values[($numOfSections-1)] ne "info" &&
          $values[($numOfSections-1)] ne "name" &&
          $values[($numOfSections-1)] ne "pro" &&
          $values[($numOfSections-1)] ne "museum" &&
          $values[($numOfSections-1)] ne "aero" &&
          $values[($numOfSections-1)] ne "coop" &&
          length($values[($numOfSections-1)]) != 2
	) {
        return("INVALID_SITENAME_BAD_SUFFIX");
    }
    else {
        return("");
    }
 
}

# A sub-routine to verify that a address is valid for a Pinging
sub validPing {
 
    my $sitename = lc shift || return;
 
    my @values = split('\.', $sitename);
    my $numOfSections = scalar(@values);
 
    my $sectionTooLong = 0;
    foreach my $value (@values) {
        if (length($value) > 63) {
            $sectionTooLong = 1;
        }
    }
 
    # Check for spaces
    if ($sitename =~ m/ /) {
        return("INVALID_SITENAME_HASSPACES");
    }
    # Check for invalid characters
    elsif ($sitename =~ m/[^\dA-Za-z\.\-]/) {
        return("INVALID_SITENAME_INVALIDCHARACTERS");
    }
    # Check for a digit at the beginning
    #elsif ($sitename =~ m/^\d/) {
    #    return("INVALID_SITENAME_BEGINSWITHDIGIT");
    #}
    # Check for proper length between the dots
    elsif ($sectionTooLong) {
        return("INVALID_SITENAME_SECTION_TOOLONG");
    }
    # Is It Blank
    elsif ($sitename eq "") {
        return("NFSSTRING_BLANK");
    }
    else {
        return("");
    }
}

# A sub-routine to verify that a address is valid for a Static Host
sub validStaticHost {
 
    my $sitename = lc shift || return;
 
    my @values = split('\.', $sitename);
    my $numOfSections = scalar(@values);
 
    my $sectionTooLong = 0;
    foreach my $value (@values) {
        if (length($value) > 63) {
            $sectionTooLong = 1;
        }
    }
 
    # Check for spaces
    if ($sitename =~ m/ /) {
        return("INVALID_STATICHOST_HASSPACES");
    }
    # Check for invalid characters
    elsif ($sitename =~ m/[^\dA-Za-z\.\-]/) {
        return("INVALID_STATICHOST_INVALIDCHARACTERS");
    }
    # Check for a digit at the beginning
    elsif ($sitename =~ m/^\d/) {
        return("INVALID_STATICHOST_BEGINSWITHDIGIT");
    }
    # Check for proper length between the dots
    elsif ($sectionTooLong) {
        return("INVALID_STATICHOST_SECTION_TOOLONG");
    }
    # Is It Blank
    elsif ($sitename eq "") {
        return("STATICHOST_BLANK");
    }
    else {
        return("");
    }
}

# A sub-routine to verify that a domainname is valid
sub validDomainName {
 
    my $domainname = lc shift || return;
 
    my @values = split('\.', $domainname);
    my $numOfSections = scalar(@values);
 
    my $sectionTooLong = 0;
    foreach my $value (@values) {
        if (length($value) > 63) {
            $sectionTooLong = 1;
        }
    }
 
    my $numOfOctets = $numOfSections - 1;
    my $emptyoctet = "";
 
    # Check to see if we have blank octets
    foreach my $octet (@values) {
        if ($octet eq "") {
            $emptyoctet = 1;
        }
    }
 
    # Check for spaces
    if ($domainname =~ m/ /) {
        return("INVALID_DOMAINNAME_HASSPACES");
    }
    # Check for an extra dot
    elsif ($emptyoctet ne "") {
        return("INVALID_DOMAINNAME_EXTRADOT");
    }
    # Check for invalid characters
    elsif ($domainname =~ m/[^\dA-Za-z\.\-]/) {
        return("INVALID_DOMAINNAME_INVALIDCHARACTERS");
    }
    elsif ($numOfSections < 2) {
        return("INVALID_DOMAINNAME_BADFORMAT");
    }
    # Check for a digit at the beginning
    elsif ($domainname =~ m/^\d/) {
        return("INVALID_DOMAINNAME_BEGINSWITHDIGIT");
    }
    # Check for proper length between the dots
    elsif ($sectionTooLong) {
        return("INVALID_DOMAINNAME_SECTION_TOOLONG");
    }
    # Check for maximum length not to exceed 254
    elsif (length($domainname) > 254) {
        return("INVALID_DOMAINNAME_TOOLONG");
    }
    # Check for proper suffix (.com, .net, .org, .gov, .edu, .mil., .int, .??)
    elsif (
	   $values[($numOfSections-1)] ne "com" &&
	   $values[($numOfSections-1)] ne "net" &&
	   $values[($numOfSections-1)] ne "org" &&
	   $values[($numOfSections-1)] ne "gov" &&
	   $values[($numOfSections-1)] ne "edu" &&
	   $values[($numOfSections-1)] ne "mil" &&
	   $values[($numOfSections-1)] ne "int" &&
	   $values[($numOfSections-1)] ne "biz" &&
	   $values[($numOfSections-1)] ne "info" &&
	   $values[($numOfSections-1)] ne "name" &&
	   $values[($numOfSections-1)] ne "pro" &&
	   $values[($numOfSections-1)] ne "museum" &&
	   $values[($numOfSections-1)] ne "aero" &&
	   $values[($numOfSections-1)] ne "coop" &&
	   length($values[($numOfSections-1)]) < 2
	   
	   ) {
        return("INVALID_DOMAINNAME_BAD_SUFFIX");
    }
    elsif ($domainname =~ m/\.$/) {
        return("INVALID_DOMAINNAME_BAD_SUFFIX2");
    }
    else {
        return("");
    }
    
}

# A sub-routine to verify that an IP address is valid
sub validIP {
    
    my $ip = shift || return;
    my $netmask = shift || "";
    
    my @netmasks = split(".", $netmask);
    
    # Check for on-digit characters
    if ($ip =~ m/ /) {
        return("INVALID_IP_HASSPACES");
    }
    elsif ($ip =~ m/[^\d\.]/) {
        return("INVALID_IP_HASNONDIGITS");
    }
    elsif (!($ip =~ m/^(\d{1,3})\.(\d{1,3})\.(\d{1,3})\.(\d{1,3})$/) ) {
        return("INVALID_IP");
    }
    elsif ($1 > 255 || $2 > 255 || $3 > 255 || $4 > 255) {
        return("INVALID_IP_OUTOFRANGE");
    }
    elsif ($4 == 255) {
        return ("INVALID_IP_BROADCAST");
    }
    elsif ($1 == 127) {
        return ("INVALID_IP_LOOPBACK");
    }
    elsif ($1 == 255 && $2 == 255 && $3 == 255 && $4 == 255) {
        return ("INVALID_IP_BROADCAST");
    }
    elsif ($1 == 0) {
        return ("INVALID_IP");
    }
    elsif ($1 == 0 && $2 == 0 && $3 == 0 && $4 == 0) {
        return ("INVALID_IP");
    }
    # If we have netmask, check against that as well
    elsif ($netmask ne "") {
	
        if ($netmask eq "255.255.255.0" && $4 == 0) {
            return ("INVALID_IP");
        }
        elsif ($netmask eq "255.255.0.0" && $3 == 0 && $4 == 0) {
            return ("INVALID_IP");
        }
        elsif ($netmask eq "255.0.0.0" && $2 == 0 && $3 == 0 && $4 == 0) {
            return ("INVALID_IP");
        }
	
    }
    elsif ($4 == 0) {
        return ("INVALID_IP");
    }
    elsif ($1 > 223) {
        return ("INVALID_IP_MULTICAST");
    }
    else {
        return("");
    }
    
    return("");
    
} 

# A sub-routine to verify that a Network is valid
sub validNetwork {
 
    my $network = shift || return;
 
    # Check for non-digit characters
    if ($network =~ m/ /) {
        return("INVALID_NETWORK_HASSPACES");
    }
    elsif ($network =~ m/[^\d\.]/) {
        return("INVALID_NETWORK_HASNONDIGITS");
    }
    elsif (!($network =~ m/^(\d{1,3})\.(\d{1,3})\.(\d{1,3})\.(\d{1,3})$/) ) {
        return("INVALID_NETWORK");
    }
    elsif ($1 > 255 || $2 > 255 || $3 > 255 || $4 > 255) {
        return("INVALID_NETWORK_OUTOFRANGE");
    }
    elsif ($1 == 0) {
        return("INVALID_NETWORK");
    }
    else {
        return("");
    }
}

# A sub-routine to verify that a Net Mask is valid
sub validNetMask {
 
    my $netmask = shift || return;
 
    my $goodform = $netmask =~ m/^(\d{1,3})\.(\d{1,3})\.(\d{1,3})\.(\d{1,3})/;
 
    print STDERR "Netmaks octets = $1 $2 $3 $4\nGoodForm = $goodform\n";
 
    # Check for non-digit characters
    if ($netmask =~ m/ /) {
        return("INVALID_NETMASK_HASSPACES");
    }
    elsif ($netmask =~ m/[^\d\.]/) {
        return("INVALID_NETMASK_HASNONDIGITS");
    }
    elsif (!$goodform) {
        return("INVALID_NETMASK");
    }
    elsif ($1 > 255 || $2 > 255 || $3 > 255 || $4 > 255) {
        return("INVALID_NETMASK_OUTOFRANGE");
    }
    elsif ($1 == 0) {
        return ("INVALID_NETMASK");
    }
    elsif ($1 < 255 && ($2 != 0 || $3 != 0 || $4 !=0)) {
        print STDERR "Problem with the second octet\n";
        return ("INVALID_NETMASK");
    }
    elsif ($1 == 255 && $2 < 255 && ($3 != 0 || $4 != 0)) {
        print STDERR "Problem with the third octet\n";
        return ("INVALID_NETMASK");
    }
    elsif ($1 == 255 && $2 == 255 && $3 < 255 && $4 !=0) {
        return ("INVALID_NETMASK");
        print STDERR "Problem with the fourth octet\n";
    }
    else {
        return("");
    }
 
}

# A sub-routine to verify that a Gateway is valid
sub validGateway {
 
    my $ip = shift || return;
    my $netmask = shift || "";
 
    my @netmasks = split(".", $netmask);
 
    # Check for on-digit characters
    if ($ip =~ m/ /) {
        return("INVALID_GW_HASSPACES");
    }
    elsif ($ip =~ m/[^\d\.]/) {
        return("INVALID_GW_HASNONDIGITS");
    }
    elsif (!($ip =~ m/^(\d{1,3})\.(\d{1,3})\.(\d{1,3})\.(\d{1,3})$/) ) {
        return("INVALID_GW");
    }
    elsif ($1 > 255 || $2 > 255 || $3 > 255 || $4 > 255) {
        return("INVALID_GW_OUTOFRANGE");
    }
    elsif ($4 == 255) {
        return ("INVALID_GW_BROADCAST");
    }
    elsif ($1 == 127) {
        return ("INVALID_GW_LOOPBACK");
    }
    elsif ($1 == 255 && $2 == 255 && $3 == 255 && $4 == 255) {
	return ("INVALID_GW_BROADCAST");
    }
    elsif ($1 == 0) {
        return ("INVALID_GW");
    }
    elsif ($1 == 0 && $2 == 0 && $3 == 0 && $4 == 0) {
        return ("INVALID_GW");
    }
    # If we have netmask, check against that as well
    elsif ($netmask ne "") {
 
        if ($netmask eq "255.255.255.0" && $4 == 0) {
            return ("INVALID_GW");
        }
 
        elsif ($netmask eq "255.255.0.0" && $3 == 0 && $4 == 0) {
            return ("INVALID_GW");
        }

        elsif ($netmask eq "255.0.0.0" && $2 == 0 && $3 == 0 && $4 == 0) {
            return ("INVALID_GW");
        }
    }
    elsif ($4 == 0) {
        return ("INVALID_GW");
    }
    elsif ($1 > 223) {
        return ("INVALID_GW_MULTICAST");
    }
    else {
        return("");
    }
 
    return("");
 
}

# A sub-routine to verify that a Login Name is valid
sub validPassword {
 
    my $password = shift || return(0);
 
    # Check for spaces
    if ($password =~ m/ /) {
        return("PASSWORD_HASSPACES");
    }
    # Check for invalid characters
    elsif ($password =~ m/[%&;\-=+]/) {
        return("PASSWORD_INVALIDCHARACTERS");
    }
    if (length($password) < 8) {
        return("PASSWORD_TOOSHORT");
    }
 
    return("");
 
}

# A sub-routine to verify that a Login Name is valid
sub validLoginName {
 
    my %system = config::system();
 
    my $loginname = shift || return(0);
    my $parser = new XML::DOM::Parser;
    my $doc = $parser->parsefile("$system{'datastore'}/general.xml");
 
    my $admin = $doc->getElementsByTagName("Admin")->item(0)->getElementsByTagName("LoginName")->item(0)->getFirstChild->getData();
 
    # Check for spaces
    if ($loginname =~ m/ /) {
        return("LOGINNAME_HASSPACES");
    }
    # Check for invalid characters
    elsif ($loginname =~ m/[^\w\.]/) {
        return("LOGINNAME_INVALIDCHARACTERS");
    }
    # Check for numbers
    elsif ($loginname =~ m/^\d/) {
        return("LOGINNAME_BEGINSWITHDIGIT");
    }

    return("");
 
}

# A sub-routine to verify that an NFS connect string is valid
sub validNFSConnectString {
 
    my $nfsstring = shift || return;
    my @values = split(':', $nfsstring);
 
    my $error = "";
 
    # The NFS connect string must have exactly two components
    if (scalar(@values) != 2) {
        return("NFSSTRING_BADFORMAT");
    }
    # Check to make sure the second one is a valid host name
    if ($error = valid::validIP($values[0])) {
        if ($error = valid::validDomainName($values[0])) {
            return($error);
        }
    }
 
    # Make sure there is a "/" at the beginning
    if (!($values[1] =~ m/^\//)) {
        return("NFSSTRING_BADFORMAT");
    }
 
    # Is it blank?
    if ($values[1] eq "") {
        return("NFSSTRING_BLANK");
    }
 
    # Looks good
    return("");
 
}

1;
