package system;
#
#  Package:      system.pm
#
#  Copyright:     COPYRIGHT Compaq Computer Corporation 2001.  All rights reserved.
#
#  Description:  The OS specific routines for changing the system.
#                These routines handle basic system functions
#
#  History:
#
use strict;

use CGI;
use XML::DOM;
use XML::QL;
use Quota;
use Time::localtime;

use config;
use general;
use datastore;
use message;
use error;

# A sub-routine to update the system date and time
sub updateDateTime {

    my $mode = shift || return(0);
    my $datetime = shift || return(0);
    my $timezone = shift || "";
    my $timeserver = shift || "";
    my $frequency = shift || "No";

    my %system = config::system();

    # Update the time zone file
    if ($timezone ne "") {
	# Copy the right time zone file over
	`/bin/cp $timezone /etc/localtime`;	
    }

    # Use a manual update
    my $command = "";
    my $result = "";
    my $errors = "";

    # Update via a time server and setup time synch
    if ($mode eq "ntp") {
	
	eval {
	    
	    # Set error handling
	    local $SIG{ALRM} = sub {
		my $errormsg = error::throwError("NTP_TIMEOUT"); 
		print "NTP_TIMEOUT";
		warn("$errormsg");
		die("$errormsg");
	    };
	    
	    # Set timeout
	    alarm(($system{'ntptimeout'}));

	    open(STDERR, ">/tmp/ntpdateresults.tmp");
	    $| = 1;
	    $command = "/usr/sbin/ntpdate -t1 $timeserver";
	    general::debug( "Making NTPDATE call = $command");
	    $result = `$command`;

	    if (open (ERRORS, "</tmp/ntpdateresults.tmp")) {
		
		my @errorlines = <ERRORS>;

		$errors = join("\n",@errorlines);
		
		close(ERRORS);
	    }

	};

#	general::debug( "Results = $result");

	# Check to see if ntpdate timed out
	if ($errors =~ m/no server suitable/) {
	    print "NTP_NOSERVICE";
	    exit(0);
	}
	elsif ($errors =~ m/can\'t find host/) {
	    print "NTP_NOHOST";
	    exit(0);
	}
	# Print the date/time that was returned
	else {
	    my @values = split("ntpdate", $result);
	    print "$values[0]\n";
}

    }
    # Update manually
    else {
	$command = "$system{'root'}/system/bin/date --set='$datetime'";
	$result = `$command`;	
    }

    # Setup time synchronization to occur (or not occur) on a routine basis
    # Grab current crontab
    $command = "/usr/bin/crontab -l";
    $result = `$command`;

    $result =~ s/\"/\\\"/g;

#    general::debug( "CRONTAB = $result");
    
    my @lines = split('\n', $result);
    my @newlines = ();
    
    my $numOfLines = scalar(@lines);
    
    for (my $i=3; $i<$numOfLines; $i++) {
	
	# Remove the time synch lines
	if ($lines[$i] =~ /# Compaq Configurator/) {
	    next;
	}
	elsif ($lines[$i] =~ m/\/system\/bin\/ntpdate/) {
	    next;
	}
	# Get rid of blank lines
	elsif ($lines[$i] eq "\n") {
	    next;
	}
	# Preserve existing entries
	else {
	    push(@newlines, $lines[$i]);
	}
    }

    # Add time synch lines
    if ($frequency ne "No") {
	push (@newlines, "\n# Compaq Configurator - Time Synchronization");
    }
    
    if ($frequency eq "hour") {
	push (@newlines, "0 * * * * $system{'root'}/system/bin/ntpdate -s -t 3 $timeserver");
    }
    elsif ($frequency eq "day") {
	push (@newlines, "0 0 * * * $system{'root'}/system/bin/ntpdate -s -t 3 $timeserver");
    }
    elsif ($frequency eq "week") {
	push (@newlines, "0 0 * * 0 $system{'root'}/system/bin/ntpdate -s -t 3 $timeserver");
    }
    elsif ($frequency eq "month") {
	push (@newlines, "0 0 1 * * $system{'root'}/system/bin/ntpdate -s -t 3 $timeserver");
    }
    
    $command = "echo \"";
    
    foreach my $line (@newlines) {
	$command .= "$line\n";
    }
    
    $command .="\" | /usr/bin/crontab -";

    $result = `$command`;

    return(1);

}

# A sub-routine to shut the system down
sub shutdown {

    my $postAction = shift || "";
    my $when = shift || "";

    my $time = "";
    if ($when eq "now") {
	$time = "now";
    }

    my $command = "";

    # Determine the type of shutdown
    if ($postAction eq "halt") {
	$command = "/sbin/shutdown -h $time";
    }
    else {
	$command = "/sbin/shutdown -r $time";
    }

    # Do the shutdown
    `$command`;

    return (1);

}

# A sub-routine to update the root password
sub updateRoot {

    my $password = shift || return;

    # Encrypt password
    $password = crypt($password, substr($password,0,2));

    my $command = "/usr/sbin/usermod -p $password root";
    print "$command\n";
    print `$command`;

    return(1);

}
# A sub-routine to update log configurations
sub updateLogConfig {

    general::debug( "Calling updateLogConfig in custom");

    my %system = config::system();

    # Step 1:  Parse information about log rotation from XML
    my $parser = new XML::DOM::Parser;
    my $doc = $parser->parsefile("$system{'datastore'}/general.xml");

    my $date = `/bin/date`;

    # Grab the values for the log files
    my $usesyslogdserver = eval { $doc->getElementsByTagName("Logs")->item(0)->getElementsByTagName("UseSyslogdServer")->item(0)->getFirstChild->getData(); } || "";
    my $syslogdserver = eval { $doc->getElementsByTagName("Logs")->item(0)->getElementsByTagName("SyslogdServer")->item(0)->getFirstChild->getData(); } || "";
    my $syslog = eval { $doc->getElementsByTagName("Logs")->item(0)->getElementsByTagName("Syslog")->item(0)->getFirstChild->getData(); } || "";
    my $weblog = eval { $doc->getElementsByTagName("Logs")->item(0)->getElementsByTagName("Weblog")->item(0)->getFirstChild->getData(); } || "";
    my $maillog = eval { $doc->getElementsByTagName("Logs")->item(0)->getElementsByTagName("Maillog")->item(0)->getFirstChild->getData(); } || "";

    my $weblogfile = "
/var/log/httpd/access_log {
    missingok
    postrotate
        /bin/kill -HUP `cat /var/run/httpd.pid 2>/dev/null` 2> /dev/null || true
    endscript
}

/var/log/httpd/agent_log {
    missingok
    postrotate
        /bin/kill -HUP `cat /var/run/httpd.pid 2>/dev/null` 2> /dev/null || true
    endscript
}

/var/log/httpd/error_log {
    missingok
    postrotate
        /bin/kill -HUP `cat /var/run/httpd.pid 2>/dev/null` 2> /dev/null || true
    endscript
}

/var/log/httpd/referer_log {
    missingok
    postrotate
        /bin/kill -HUP `cat /var/run/httpd.pid 2>/dev/null` 2> /dev/null || true
    endscript
}
";

    my $syslogfile = "/var/log/messages {
    postrotate
/bin/kill -HUP `cat /var/run/syslogd.pid 2> /dev/null` 2> /dev/null || true
    endscript
}

/var/log/secure {
    postrotate
/bin/kill -HUP `cat /var/run/syslogd.pid 2> /dev/null` 2> /dev/null || true
    endscript
}

/var/log/maillog {
/bin/kill -HUP `cat /var/run/syslogd.pid 2> /dev/null` 2> /dev/null || true
    endscript
}

/var/log/spooler {
    postrotate
/bin/kill -HUP `cat /var/run/syslogd.pid 2> /dev/null` 2> /dev/null || true
    endscript
}

/var/log/boot.log {
    postrotate
/bin/kill -HUP `cat /var/run/syslogd.pid 2> /dev/null` 2> /dev/null || true
    endscript
}

/var/log/cron {
    postrotate
/bin/kill -HUP `cat /var/run/syslogd.pid 2>/dev/null` 2>/dev/null || true
    endscript
}
";

    # weblog
    if ($weblog eq "Once a Month") {
	$weblogfile =~ s/missingok/missingok\n     monthly/g;
    }
    elsif ($weblog eq "When greater than 5 MB") {
	$weblogfile =~ s/missingok/missingok\n     size=5M/g;
    }
    elsif ($weblog eq "When greater than 10 MB") {
	$weblogfile =~ s/missingok/missingok\n     size=10M/g;
    }
    elsif ($weblog eq "When greater than 50 MB") {
	$weblogfile =~ s/missingok/missingok\n     size=50M/g;
    }
    else {
	$weblogfile =~ s/missingok/missingok\n     weekly/g;
    }

    # syslog
    if ($syslog eq "Once a Month") {
	$syslogfile =~ s/postrotate/monthly\n     postrotate/g;
    }
    elsif ($syslog eq "When greater than 5 MB") {
	$syslogfile =~ s/postrotate/size=5M\n     postrotate/g;
    }
    elsif ($syslog eq "When greater than 10 MB") {
	$syslogfile =~ s/postrotate/size=10M\n     postrotate/g;
    }
    elsif ($syslog eq "When greater than 50 MB") {
	$syslogfile =~ s/postrotate/size=50M\n     postrotate/g;
    }
    else {
	$syslogfile =~ s/postrotate/weekly\n     postrotate/g;
    }

    # maillog
    if ($maillog eq "Once a Month") {
	$syslogfile =~ s/\/var\/log\/maillog {/\/var\/log\/maillog {\n    postrotate\n    monthly\n/;
    }
    elsif ($maillog eq "When greater than 5 MB") {
	$syslogfile =~ s/\/var\/log\/maillog {/\/var\/log\/maillog {\n    postrotate\n    size=5M\n/;
    }
    elsif ($maillog eq "When greater than 10 MB") {
	$syslogfile =~ s/\/var\/log\/maillog {/\/var\/log\/maillog {\n    postrotate\n    size=10M\n/;
    }
    elsif ($maillog eq "When greater than 50 MB") {
	$syslogfile =~ s/\/var\/log\/maillog {/\/var\/log\/maillog {\n    postrotate\n    size=50M\n/;
    }
    else {
	$syslogfile =~ s/\/var\/log\/maillog {/\/var\/log\/maillog {\n    postrotate\n    weekly\n/;
    }

    # Step 2:  Build new log rotation configuration file
    my $newlogconfig = "
# Modified by Compaq Configurator - $date

# see \"man logrotate\" for details
# rotate log files weekly
daily

# keep 4 weeks worth of backlogs
rotate 4

# send errors to root
errors root

# create new (empty) log files after rotating old ones
create

# uncomment this if you want your log files compressed
compress

# RPM packages drop log rotation information into this directory
include /etc/logrotate.d

# no packages own lastlog or wtmp -- we'll rotate them here
/var/log/wtmp {
    daily
    create 0664 root utmp
    rotate 1
}
";

    # Step 2a:  Write syslog file out
    if (open(FILE, ">/etc/logrotate.d/syslog")) {

	print FILE $syslogfile;

	close (FILE);

    }

    # Step 2b:  Write apache file out
    if (open(FILE, ">/etc/logrotate.d/apache")) {

	print FILE $weblogfile;

	close (FILE);

    }

    # Step 3:  Write the new log configuration file out
    if (open(FILE, ">/etc/logrotate.conf")) {

	print FILE $newlogconfig;

	close (FILE);

    }

    # Step 4:  Open the syslog.conf file
    if (open (FILE, "</etc/syslog.conf")) {

	general::debug( "Opened syslog.conf read only mode  usesyslogdserver=$usesyslogdserver");

	# Step 5:  Setup a syslogd server
	my @lines = <FILE>;

	close (FILE);

	# Step 6:  Write the file back out
	if (open (FILE, ">/etc/syslog.conf")) {

	    foreach my $line (@lines) {

		# Modify the syslog line to log to a server if requested
		if ($line =~ m/^\*\.info;/ && $usesyslogdserver eq "Yes") {
		    general::debug( "Matched info line and changing");
		    $line = "*.info;mail.none;authpriv.none                          \@$syslogdserver\n";
		}
		elsif ($line =~ m/^\*\.info;/) {
		    $line = "*.info;mail.none;authpriv.none                          /var/log/messages\n";
		}

		print FILE "$line";

	    }

	    close(FILE);

	}

    }

    general::debug( "Trying to open cpqcfg log rotation config in write mode");

    # Re-build the cpqcfg log rotation config. file
    if(open(LOGFILE, ">/etc/logrotate.d/cpqcfg")) {

	general::debug( "Writing out cpqcfg log rotation config file");

	print LOGFILE "
# cpqcfg log rotation
/var/cpqcfg/httpd/log/access_log {
     missingok
     size=5M
     postrotate
	/etc/rc.d/init.d/cpqcfg stop 2> /dev/null 1> /dev/null
	/etc/rc.d/init.d/cpqcfg start 2> /dev/null 1> /dev/null
     endscript
}

/var/cpqcfg/httpd/log/error_log {
     missingok
     size=5M
     postrotate
	/etc/rc.d/init.d/cpqcfg stop 2> /dev/null 1> /dev/null
	/etc/rc.d/init.d/cpqcfg start 2> /dev/null 1> /dev/null
     endscript
}
";

	close(LOGFILE);

    }

}
# A sub-routine to set the current configuration status once rapid startup has been run
sub setConfigStatus {

    my @lines = ();

    # Read the lines in

    if (open(FILE, "</var/www/html/index.html")) {

        @lines = <FILE>;
	close(FILE);
    }

    if (open(FILE, ">/var/www/html/index.html")) {
	
	foreach my $line(@lines) {
	    
	    $line =~ s/<META NAME="cpqstatus" CONTENT="1">//g;
	    print FILE $line;
	}

	close(FILE);
    }
}
1;
