################################################################################
# Copyright (c) 2001 - Compaq Computer Corporation - All rights reserved
# Compaq Confidential
################################################################################
#
# Module:  $RCSfile: cpqpwrmgr.tcl,v $
#
# Version: $Revision: 1.4 $
#
# Author:  $Author: jlong $
#
# Date:    $Date: 2001/04/03 20:15:40 $
################################################################################

# Load message source file.
set GUI_PATH [exec dirname $argv0]
source $GUI_PATH/pwrmgr.msg.tcl

# Global variables
set msg(0) "Gracefully shutdown the OS before turning off power."
set msg(1) "Turn power off to machine. (Normal)"
set msg(2) "Do Nothing. (Power Switch is disabled)"
set cmd(0) "GracefulCB"
set cmd(1) "TypicalCB"
set cmd(2) "DisableCB"
set Graceful 3
set Disable  2
set Typical  1
set state $Typical
set tstate $state
set MIN 1
set MAX 120
set DEFAULT 10
set delay $DEFAULT
set tdelay $DEFAULT
set SAVED	1

# System Specifics
set hostName [exec hostname]
set EXEC_PROG	"/sbin/pwrmgrBE"
set LOGO "/usr/share/pixmaps/compred.xbm"

#+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# Procedure: SetChanged
#
# Purpose:   Changes selected delay to 1 second (Minimum delay) if / when
#	     the user selects Disable the Power Switch option.
#
# Input:     None, uses global variables.
#
# Return:    None, changes the value of the temporary delay variable, tdelay,
#            and the state of the "Save" button.
#-------------------------------------------------------------------------------
proc SetChanged { args } {

	global SAVED tstate tdelay Disable scl f

	if { $tstate == $Disable } {
		set tdelay 1
		$scl set $tdelay
	}
	set SAVED 0
	$f.save configure -state normal
}

#+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# Procedure:  ShowChoices
#
# Purpose:    Build and display the set of RadioButtons to provide the user a
#             means of selecting what action should occur when the Power
#             switch is pressed.
#
# Inputs:     parent - path to parent window.
#             varname - the name of the variable to contain the value of the 
#	                selected button, 'tstate'
#             args - the number and name of each button to be displayed.
#
# Returns:   varname - value of the selected button. In this case, 'tstate'
#
#-------------------------------------------------------------------------------
proc ShowChoices { parent varname args } {

	global msg cmd tstate bts PWRMGR_MSGS

	set rb [frame $parent.choices -height 200 -borderwidth 2 -relief groove]
	label $rb.label1 -text $PWRMGR_MSGS(MSG_LBL_BEHAVIOR)
	label $rb.label2 -text $PWRMGR_MSGS(MSG_TXT_ACTION)
	pack $rb.label1 -anchor w  -padx 3
	pack $rb.label2 -anchor w -padx 20 -pady 10
	set bts [frame $rb.btns \
		-width 200 \
		-borderwidth 2 \
		-relief sunken]
	set b 0
	foreach item $args {
		radiobutton $bts.$b \
			-text $msg($b) \
			-command SetChanged \
			-variable $varname \
			-selectcolor maroon4 \
			-takefocus 1 \
			-value $b \
			-cursor hand1
		pack $bts.$b -side top -anchor w -padx 30 -pady 3
		incr b
	}
	pack $bts -expand true \
		-anchor w \
		-padx 30 \
		-pady 5 \
		-fill x
	pack $rb -expand true \
		-anchor nw \
		-fill x \
		-pady 3 \
		-padx 3
}

#+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# Procedure:  ShowScale
#
# Purpose:    Builds and displays a horizontal sliding scale for the user to
#             select the amount of delay in seconds, the desire to elapse
#             before the selected power switch behavior occurs.
#
# Inputs:     parent - path name of parent window.
#             varname - Variable to contain the selected value, 'tdelay'
#
# Returns:    varname - or 'tdelay'
#
#-------------------------------------------------------------------------------
proc ShowScale { parent varname args } {
	
	global MIN MAX DEFAULT tdelay scl PWRMGR_MSGS

	set sc [frame $parent.scale \
		-borderwidth 2 \
		-relief groove]
	label $sc.label \
		-text $PWRMGR_MSGS(MSG_LBL_DELAY)
	label $sc.label1 \
		-text $PWRMGR_MSGS(MSG_TXT_DELAY)
	pack $sc.label -anchor nw -padx 3
	pack $sc.label1 -anchor w -padx 20 -pady 5

	set scl [scale $sc.scale1 -borderwidth 2 \
		-relief flat  \
		-from $MIN \
		-to $MAX \
		-length 200 \
		-orient horizontal \
		-troughcolor maroon4 \
		-bigincrement 0 \
		-resolution 1 \
		-takefocus 1 \
		-command GetDelayCB \
		-label $PWRMGR_MSGS(MSG_LBL_SECS) \
		-cursor hand2 \
		-showvalue true \
		-variable $tdelay]

	pack $scl -side top -anchor w -fill x -pady 10 -padx 30
	pack $sc -expand true -side top -anchor w -fill x -pady 3 -padx 3
	$scl set $tdelay
}

#+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# Procedure:  GetDelayCB
#
# Purpose:    Callback proc for the delay scale widget.  Captures the new value
#             and stuffs it into the 'tdelay' variable.  Also flags the SAVED
#             variable to indicate changed data, and sets state of the "Save"
#             button to normal, or active.
#
# Inputs:     None
#
# Returns:    tdelay, SAVED
#
#-------------------------------------------------------------------------------
proc GetDelayCB { args } {

	global tdelay delay scl SAVED f

	set tdelay [$scl get]
	if { $tdelay != $delay } {
		set SAVED 0
		$f.save configure -state normal
	}
}

#+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# Procedure:  SaveCB
#
# Purpose:    Writes the selected options and delay to the system EV.  Checks
#             and validates the write operation succeeded, as best we can, and
#             resets the delay, state, tdelay, tstate variables in agreement.
#             Also resets the "Save" button to disabled.
#
# Inputs:     tdelay, tstate, SAVED
#
# Returns:    tdelay, tstate, delay, state, SAVED
#
#-------------------------------------------------------------------------------
proc SaveCB { args } {

	global EXEC_PROG delay tdelay state tstate SAVED Graceful f

	if { $tstate == 0 } {
		set state $Graceful
	} else {
		set state $tstate
	}
	set delay $tdelay
	set input [exec echo "2* $state* $delay* ." | $EXEC_PROG]
	set response [split $input *]

	if { [lindex $response 0] == "ERROR: " } {
		DisplayError "." "$response"
	}
	set SAVED 1
	$f.save configure -state disabled
}

#+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# Procedure:  GetEV
#
# Purpose:    Validates that the following:
#                 User is 'superuser'
#                 cpqhealth driver is installed and loaded.
#             then, accesses the binary backend program to retrieve any values
#             of delay, and switch state, that may be stored in the System EV.
#
# Inputs:     delay, state, tdelay, tstate
#
# Returns:    If successful - delay, state, tdelay, tstate.
#             On ERROR - Calls a proc to display the returned ERROR message,
#                        then exits.
#
#-------------------------------------------------------------------------------
proc GetEV { args } {

	global pipe EXEC_PROG delay tdelay state tstate PWRMGR_MSGS

	set response ""
	set errormsg ""
	set rest ""

# Catch an error from Backend binary, should one occur!
	if [ catch {exec echo "1* * *" | $EXEC_PROG } input] {
		regsub "$PWRMGR_MSGS(MSG_ERR_EXIT)" $input " " errormsg
		DisplayError "." $errormsg
	}
	set response [split $input *]

	regexp {ERROR: } $input errormsg
	regexp {^.*\.} $input rest

# Double check for errors.
	if { "$errormsg" == "ERROR: " } {
		DisplayError "." "$rest"
	} else {
# Otherwise proceed normally.
		set delay [lindex $response 0]
		set state [lindex $response 1]
		set tdelay $delay
# Required to match displayed location of RadioButton with Health reqmts for
# Switch state.
		switch -regexp -- $state { 

			"1" { set tstate 1 }

			"2" { set tstate 2 }

			"3" { set tstate 0 }
		}
	}
}

#+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# Procedure:  QuitCB
#
# Purpose:    To close the application and exit.  First makes a check to see if
#             the user has changed any data, but not yet saved it.  If so, it
#             makes a call to a proc to display a message advising the user
#             of same.
#
# Inputs:     variable SAVED
#
# Returns:    exit program
#
#-------------------------------------------------------------------------------
proc QuitCB { args } {

	global SAVED

	if { ! $SAVED } {
		DisplayQuitMsg
	} else {
		exit
	}
}

#+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# Procedure:  DisplayQuitMsg
#
# Purpose:    Display a Dialog window containing a message advising the user
#             they have not yet saved data modifications, and offers them the
#             option of saving it.
#
# Inputs:     None
#
# Returns:    If user selects:
#                 YES - Calls SaveCB, then exits.
#                  NO - Exits
#              CANCEL - Cancels the message, and returns to the application.
#
#-------------------------------------------------------------------------------
proc DisplayQuitMsg { args } {

	global PWRMGR_MSGS
	set choice [tk_messageBox \
		-type yesnocancel \
		-default yes \
		-title $PWRMGR_MSGS(MSG_DIALOG_TITLE) \
		-message $PWRMGR_MSGS(MSG_DIALOG_NOSAVE) \
		-icon question]

	switch -- $choice {
		yes { SaveCB; exit }
		cancel { return }
		default { exit }
	}
}

#+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# Procedure:  DisplayError
#
# Purpose:    Displays an ERROR Dialog window advising user an error has
#             occurred, then exit the application.
#
# Inputs:     The message to be displayed.
#
# Returns:    None, just exit the application.
#
#-------------------------------------------------------------------------------
proc DisplayError { parent msg } {

	wm withdraw $parent
	set choice [tk_messageBox \
		-type ok \
		-default ok \
		-message $msg \
		-title "ERROR" \
		-icon error]

	switch -- $choice {
		default { exit }
	}
}

#+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# MAIN Function
#
# Validates that the Compaq Power Down Manager feature is supported by this
# hardware, and the user is logged in as superuser, or root.  Then, if so,
# displays the main window with the available selections for the user to make.
# If the feature is NOT supported, the user is not superuser, or an error
# occurs, calls a proc to display a Dialog window advising the user of same.
#-------------------------------------------------------------------------------
global PWRMGR_MSGS

# Verify user is 'superuser'
set uid ""
set user [exec id]
set uid [lindex $user 0]
set suser [string last "root" "$uid"]
if { $suser == -1 } {
	DisplayError "." $PWRMGR_MSGS(MSG_ERR_NOTROOT)
	exit
}

# call a proc to check to see if this system supports the Power Down Manager
# feature, and whether there are stored parameters in the EV.

GetEV

# Create toplevel window.
wm minsize . 30 5
wm title . "$PWRMGR_MSGS(MSG_APP_TITLE)"
wm geometry . 600x400+100+100

# Create a frame for HostName
set w [frame .frame1 -height 25 -relief groove -bd 2]
pack $w -fill x -anchor nw -pady 3 -padx 3
label $w.host -text $PWRMGR_MSGS(MSG_LBL_SYSTEM)
label $w.hostname -text $hostName
pack $w.host $w.hostname -pady 2 -padx 5 -side left

# Call proc to create a frame to hold the Behavior selections and build 
# RadioButtons.
ShowChoices {} tstate graceful typical disable

# Call proc to create a frame to hold the slider scale for delay selection.
ShowScale {} $MIN $MAX $delay

# Create a row of buttons along the bottom and build image of Compaq logo
global f
set lg [image create bitmap -file $LOGO -foreground red] 
set f [frame .menubar -height 5 -relief raised]
pack $f -fill x
button $f.exit \
	-text $PWRMGR_MSGS(MSG_BTN_EXIT) \
	-takefocus 1 \
	-command QuitCB
button $f.save \
	-text $PWRMGR_MSGS(MSG_BTN_SAVE) \
	-takefocus 1 \
	-state disabled \
	-command SaveCB
button $f.quit \
	-text $PWRMGR_MSGS(MSG_BTN_CANCEL) \
	-takefocus 1 \
	-command exit
# Setup and display Compaq logo on same row as buttons.
button $f.logo -image $lg  -relief flat -takefocus 0
pack $f.exit $f.save $f.quit $f.logo -side left -expand true -pady 5

