package groups;
#
#  Package:      groups.pm
#
#  Copyright:     COPYRIGHT Compaq Computer Corporation 2001.  All rights reserved.
#
#  Description:  The OS specific routines for changing the system.
#
#    These routines handle GROUP specific changes including:
#
#     - Add/Delete/Edit group
#     - Disk quota management
#     - Permission management
#     - etc.
#
#  History:
#
use strict;

use CGI;
use XML::DOM;
use XML::QL;
use Quota;
use Time::localtime;
use custom::users;

use config;
use general;
use datastore;
use message;
use error;

# A sub-routine to remove a group from the system
sub delGroup {

    my $groupname = shift || return;

    my %system = config::system();

#    general::debug( "Removing group from the system");
    # Remove the user from the system
    my $command = "/usr/sbin/groupdel $groupname";
    my $result = `$command`;
#    general::debug( "Group removed from the system");

    # Double check that the group really was deleted
#    general::debug( "Double check");
    my $groupcheck = `/bin/cat /etc/group | /bin/grep ^$groupname:`;
    my $usercheck = `/bin/cat /etc/passwd | /bin/grep ^$groupname:`;
#    general::debug( "END OF Double check");

    # Check to make sure that the user wasn't trying to delete
    # a primary user group.
    if ($groupcheck eq "") {

	general::debug( "Group $groupname Deleted from the system!");

	# Update the XML
	my $parser = new XML::DOM::Parser;
	my $doc = $parser->parsefile("$system{'datastore'}/groups.xml");
	
	# Remove the group from the XML
	general::debug( "Datastore group delete");
      datastore::delGroup($doc, $groupname);
	general::debug( "END OF Datastore group delete");

	# Write out the file
	$doc->printToFile("$system{'datastore'}/groups.xml");
	
	$doc->dispose();
	
	general::debug( "Group completely deleted!");
    }
    elsif ($usercheck ne "") {
	print error::throwError("GROUP_PRIMARY_USERS","ANY",$groupname);
    }

    return(1);
}

# A sub-routine to add a group to the system
sub addGroup {

    my $shortname = shift || return("");
    my $name = shift || "";
    my $diskquota = shift || "0";
    my @members = @_;

    my %system = config::system();

    general::debug( "MEMBERS = $members[0]");

    my $groupID = "";

    # Parse the group file
    my $parser = new XML::DOM::Parser;
    my $doc = $parser->parsefile("$system{'datastore'}/groups.xml");

    # Search for this group
    my $foundNode = "";
    foreach my $groupNode ($doc->getElementsByTagName("Group")) {

	my $thisName = eval { $groupNode->getElementsByTagName("ShortName")->getFirstChild->getData(); } || "";

	if ($thisName eq $shortname) {
	    $foundNode = $groupNode;
	}
    }

    # If we didn't find the group, then add it
    if ($foundNode eq "") {

	# Add the Group
	my $command = "/usr/sbin/groupadd $shortname";
	my $result = `$command`;

	# Update the shadow file
	$result = `/usr/bin/grpconv`;

	# Grab the group ID
	$result = `/bin/cat /etc/group | /bin/grep ^$shortname:`;
	my @row = split(/:/, $result);
	$groupID = $row[2];

	# Update the XML
	my $parser = new XML::DOM::Parser;
	my $doc = $parser->parsefile("$system{'datastore'}/groups.xml");

	general::debug( "\n$groupID $shortname $name Members = @members");

        datastore::addGroup($doc, $groupID, $shortname, $name, $diskquota, @members);

	datastore::XMLtoFile($doc,"$system{'datastore'}/groups.xml");

	$doc->dispose();

	updateGroup($shortname);

    }

    return($groupID);

}

# A sub-routine to update the members of a group on the system
sub updateGroup {

    my %system = config::system();

    my $groupname = shift || return;

    general::debug( "Updating group $groupname");

    # Grab members of the group
    my $parser = new XML::DOM::Parser;
    my $doc = $parser->parsefile("$system{'datastore'}/groups.xml");

    my $groupNode = "";

    # Locate the group node for this group
    foreach my $group ($doc->getElementsByTagName("Group")) {

	my $name = eval { $group->getElementsByTagName("ShortName")->item(0)->getFirstChild->getData(); } || "";

	if ($name eq $groupname) {
	    $groupNode = $group;
	    last;
	}

    }

    # Check to make sure we found a node
    if ($groupNode eq "") {
	return;
    }

    # Open and read the group file
    my $configFile = "/etc/group";
    my @lines = ();

    # Read in the current group file
    if (open(FILE, "<$configFile")) {

	@lines = <FILE>;

	close(FILE);
    
	# Search for this group
	foreach my $line (@lines) {

	    # skip comments
	    if ($line =~ m/^#/) {
		next;
	    }
	    # else if this groups line is found
	    elsif ($line =~ m/^$groupname:/) {

		general::debug( "  *** Found the correct group to update! ($line)");

		my @values = split(':',$line);
		my @users = split(',', $values[3]);

#		general::debug( "Current members of this group $values[3]");

		# Remove all users from the group first
		foreach my $user (@users) {
		    general::debug( "    --- Removing $user");
		    users::delUserFromOSGroup($user, $groupname);
		}

		# Add the correct users back
		my $members = eval { $groupNode->getElementsByTagName("UserID"); } || "";
		my $numOfMembers = $members->getLength;

#		general::debug( "  Number of Members = $numOfMembers");
		for (my $j=0; $j < $numOfMembers; $j++) {

		    my $memberID = eval { $members->item($j)->getFirstChild->getData(); } || "";
		    my %userInfo = datastore::userInfoByID($memberID);

		    general::debug( "    ---- Adding $userInfo{'loginname'} to $groupname");

		    users::addUserToOSGroup($userInfo{'loginname'}, $groupname);
		}

		last;

	    }

	}

    }

    # Step 2 - Update group quota
    if ($groupNode ne "") {

	my $gid = $groupNode->getAttribute("id");

	my $diskquota = eval { $groupNode->getElementsByTagName("DiskQuota")->item(0)->getFirstChild->getData; } || "";

	my $dev = Quota::getqcarg("/home");
	my $diskquotablocks = $diskquota * 1.15;

	general::debug( "Updating group $gid quota $diskquota");
	general::debug( "Changed quota of $diskquota kilobytes into $diskquotablocks blocks.");

      Quota::setqlim($dev,$gid,$diskquotablocks,$diskquotablocks,0,0,0,1);

    }

    # Update shadow group
    `/usr/sbin/grpconv`;

    
}

# A sub-routine to restore all user quotas to the system
sub restoreQuotas {
 
    my %system = config::system();
 
    # Read the datastore for quotas
    my $parser = new XML::DOM::Parser;
    my $doc = $parser->parsefile("$system{'datastore'}/groups.xml");
 
    my $groups = $doc->getElementsByTagName("Group");
    my $numOfGroups = $groups->getLength;
 
  general::debug("Restoring group disk quotas!");
 
    for (my $i=0; $i<$numOfGroups; $i++) {
 
        # Get Group ID and Quota for the user
 
        my $gid = eval { $groups->item($i)->getAttribute("id"); } || "";
        my $quota = eval { $groups->item($i)->getElementsByTagName("DiskQuota")->item(0)->getFirstChild->getData(); } || "0";
 
        # Update the disk quota for this user
 
        my $dev = Quota::getqcarg("/home");
        my $diskquotablocks = $quota * 1.15;
 
      Quota::setqlim($dev,$gid,$diskquotablocks,$diskquotablocks,0,0,0,1);
 
    }

    $doc->dispose;
 
    return();
 
}

1;
